// ==========================================================
// IMPORTS
// ==========================================================
import { notificaciontoast } from "../../common/method_toast.js";
import { getButtonsSchema, languageSchema } from "../../common/schemas.js";

// ==========================================================
// VARIABLES GLOBALES
// ==========================================================
let table;
let csrfToken = $('meta[name="csrf-token"]').attr('content');
let miInstitucionId = null;

// ==========================================================
// INICIO (Document Ready)
// ==========================================================
$(document).ready(function () {
    loadDataTable();
    loadFilterInstitutions();
    initEventHandlers();
    initNewModalHandlers();
});

// ==========================================================
// FUNCIÓN: Cargar DataTable
// ==========================================================
function loadDataTable() {
    if ($.fn.DataTable.isDataTable('#datatableGeneral')) {
        table.ajax.reload(null, false);
        return;
    }

    table = $("#datatableGeneral").DataTable({
        language: languageSchema,
        processing: true,
        serverSide: false,
        ajax: {
            url: "/admin/estudiantes/traslados/list",
            type: "POST",
            data: { _token: csrfToken },
        },
        columns: [
            { data: 'nroidenti', title: 'Nro. Ident.' },
            { data: 'nombre_completo', title: 'Estudiante' },
            {
                data: 'tipo',
                title: 'Tipo',
                render: function (data) {
                    // Traducir el 0/1 que viene de la BD
                    return data == 0 ? 'Ingreso' : 'Salida';
                }
            },
            { data: 'institucion_origen_nombre', title: 'Origen' },
            { data: 'institucion_destino_nombre', title: 'Destino' },
            { data: 'fecha', title: 'Fecha' },
            {
                data: 'documento',
                title: 'Archivo',
                orderable: false,
                render: function (data, type, row) {
                    if (data) {
                        const fileName = data.split('/').pop();
                        return `<button type="button" class="btn btn-sm btn-outline-primary btn-view-document" 
                                        data-path="${data}" 
                                        data-filename="${fileName}">
                                    <i class="fas fa-file-alt"></i> Ver
                                </button>`;
                    }
                    return 'N/A';
                }
            },
            {
                data: 'id',
                title: 'Acciones',
                orderable: false,
                render: function (data, type, row) {
                    // Botón Editar
                    const editButton = `<button type="button" class="btn btn-warning shadow btn-xs sharp me-1 btn-edit-traslado" 
                                                data-id="${data}" title="Editar registro">
                                            <i class="fas fa-pencil-alt"></i>
                                        </button>`;
                    // Botón Eliminar
                    const deleteButton = `<button type="button" class="btn btn-danger shadow btn-xs sharp btn-delete-traslado"
                                                data-id="${data}" title="Eliminar registro">
                                            <i class="fas fa-trash"></i>
                                        </button>`;

                    return `<div class="d-flex justify-content-center">${editButton}${deleteButton}</div>`;
                }
            }
        ]
    });
}

// ==========================================================
// FUNCIONES PARA CARGAR SELECTS
// ==========================================================

// Carga instituciones en los FILTROS
function loadFilterInstitutions() {
    const $selectOrigen = $('#filtroInstitucionOrigen');
    const $selectDestino = $('#filtroInstitucionDestino');

    $selectOrigen.empty().append('<option value="">Cargando...</option>');
    $selectDestino.empty().append('<option value="0">(Seleccione un origen primero)</option>');
    $selectDestino.prop('disabled', true);

    const localRequest = $.ajax({
        url: '/admin/estudiantes/traslados/get-local-institution',
        type: 'GET', dataType: 'json',
    });
    const externalRequest = $.ajax({
        url: '/admin/estudiantes/traslados/get-external-institutions',
        type: 'GET', dataType: 'json',
    });

    Promise.all([localRequest, externalRequest]).then(function (results) {
        const localInst = results[0];
        const externalInsts = results[1];

        miInstitucionId = localInst.id; // Guardamos el ID local
        const localOption = `<option value="${localInst.id}">${localInst.name} (Mi Institución)</option>`;

        $selectOrigen.empty().append('<option disabled selected value="0">Filtrar por Origen</option>');
        $selectDestino.empty().append('<option disabled selected value="0">Filtrar por Destino</option>');
        $selectOrigen.append(localOption);
        $selectDestino.append(localOption);

        externalInsts.forEach(inst => {
            if (inst.id !== localInst.id) {
                const optionHtml = `<option value="${inst.id}">${inst.name}</option>`;
                $selectOrigen.append(optionHtml);
                $selectDestino.append(optionHtml);
            }
        });

        $selectOrigen.selectpicker('refresh');
        $selectDestino.selectpicker('refresh');

    }).catch(function (xhr) {
        console.error("Error al cargar listas de instituciones (Filtros):", xhr);
        notificaciontoast("error", "No se pudieron cargar las instituciones", "Error");
    });
}

// Carga instituciones en el MODAL DE CREAR (#modalNuevoTraslado)
function loadModalInstitutions() {
    const $selectOrigen = $('#institucion_origen');
    const $selectDestino = $('#institucion_destino');

    $selectOrigen.empty().append('<option value="">Cargando...</option>').selectpicker('refresh');
    $selectDestino.empty().append('<option value="">Cargando...</option>').selectpicker('refresh');

    const localRequest = $.ajax({
        url: '/admin/estudiantes/traslados/get-local-institution',
        type: 'GET', dataType: 'json',
    });
    const externalRequest = $.ajax({
        url: '/admin/estudiantes/traslados/get-external-institutions',
        type: 'GET', dataType: 'json',
    });

    Promise.all([localRequest, externalRequest]).then(function (results) {
        const localInst = results[0];
        const externalInsts = results[1];

        if (miInstitucionId === null) {
            miInstitucionId = localInst.id;
        }
        const localOption = `<option value="${localInst.id}">${localInst.name} (Mi Institución)</option>`;

        $selectOrigen.empty().append('<option disabled selected value="">Seleccione origen</option>');
        $selectDestino.empty().append('<option disabled selected value="">Seleccione destino</option>');
        $selectOrigen.append(localOption);
        $selectDestino.append(localOption);

        externalInsts.forEach(inst => {
            if (inst.id !== localInst.id) {
                const optionHtml = `<option value="${inst.id}">${inst.name}</option>`;
                $selectOrigen.append(optionHtml);
                $selectDestino.append(optionHtml);
            }
        });

        $selectOrigen.selectpicker('refresh');
        $selectDestino.selectpicker('refresh');

    }).catch(function (xhr) {
        console.error("Error al cargar instituciones (modal crear):", xhr);
        notificaciontoast("error", "No se pudieron cargar las instituciones", "Error");
    });
}

// Carga instituciones en el MODAL DE EDICIÓN (#modalEditarTraslado)
function loadModalInstitutions_Edit() {
    const $selectOrigen = $('#institucion_origen_edit');
    const $selectDestino = $('#institucion_destino_edit');

    $selectOrigen.empty().append('<option value="">Cargando...</option>').selectpicker('refresh');
    $selectDestino.empty().append('<option value="">Cargando...</option>').selectpicker('refresh');

    const localRequest = $.ajax({
        url: '/admin/estudiantes/traslados/get-local-institution',
        type: 'GET', dataType: 'json',
    });
    const externalRequest = $.ajax({
        url: '/admin/estudiantes/traslados/get-external-institutions',
        type: 'GET', dataType: 'json',
    });

    // Devolvemos la Promesa para que el listener de "Editar" sepa cuándo terminar
    return Promise.all([localRequest, externalRequest]).then(function (results) {
        const localInst = results[0];
        const externalInsts = results[1];

        if (miInstitucionId === null) {
            miInstitucionId = localInst.id;
        }
        const localOption = `<option value="${localInst.id}">${localInst.name} (Mi Institución)</option>`;

        $selectOrigen.empty().append('<option disabled selected value="">Seleccione origen</option>');
        $selectDestino.empty().append('<option disabled selected value="">Seleccione destino</option>');
        $selectOrigen.append(localOption);
        $selectDestino.append(localOption);

        externalInsts.forEach(inst => {
            if (inst.id !== localInst.id) {
                const optionHtml = `<option value="${inst.id}">${inst.name}</option>`;
                $selectOrigen.append(optionHtml);
                $selectDestino.append(optionHtml);
            }
        });

        $selectOrigen.selectpicker('refresh');
        $selectDestino.selectpicker('refresh');

    }).catch(function (xhr) {
        console.error("Error al cargar instituciones (modal edit):", xhr);
        notificaciontoast("error", "No se pudieron cargar las instituciones", "Error");
        return Promise.reject(xhr);
    });
}


// ==========================================================
// LÓGICA DE FORMULARIOS Y LISTENERS
// ==========================================================

function handleTipoTrasladoChange() {
    const tipo = $(this).val();
    const $origenSelect = $('#institucion_origen');
    const $destinoSelect = $('#institucion_destino');

    if (!miInstitucionId) {
        console.error("ID de 'Mi Institución' no encontrado.");
        return;
    }

    $origenSelect.prop('disabled', false).prop('required', false);
    $destinoSelect.prop('disabled', false).prop('required', false);
    $origenSelect.find('option').prop('disabled', false);
    $destinoSelect.find('option').prop('disabled', false);
    $origenSelect.closest('.col-md-6').slideDown();
    $destinoSelect.closest('.col-md-6').slideDown();

    if (tipo === 'Ingreso') {
        $destinoSelect.selectpicker('val', miInstitucionId);
        $destinoSelect.prop('disabled', true);
        $origenSelect.prop('required', true);
        $origenSelect.selectpicker('val', '');
        actualizarOpcionesOpuestas($destinoSelect, $origenSelect);

    } else if (tipo === 'Salida') {
        $origenSelect.selectpicker('val', miInstitucionId);
        $origenSelect.prop('disabled', true);
        $destinoSelect.prop('required', true);
        $destinoSelect.selectpicker('val', '');
        actualizarOpcionesOpuestas($origenSelect, $destinoSelect);
    } else {
        $origenSelect.closest('.col-md-6').slideUp();
        $destinoSelect.closest('.col-md-6').slideUp();
    }

    $origenSelect.selectpicker('refresh');
    $destinoSelect.selectpicker('refresh');
}
function handleTipoTrasladoChange_Edit() {
    const tipo = $(this).val();
    const $origenDiv = $('#institucion_origen_edit').closest('.col-md-6');
    const $destinoDiv = $('#institucion_destino_edit').closest('.col-md-6');

    if (tipo === 'Ingreso') {
        $origenDiv.slideDown();
        $('#institucion_origen_edit').prop('disabled', false).prop('required', true).selectpicker('refresh');

        $destinoDiv.slideUp();
        $('#institucion_destino_edit').prop('disabled', true).prop('required', false).selectpicker('val', '').selectpicker('refresh');

    } else if (tipo === 'Salida') {
        $origenDiv.slideUp();
        $('#institucion_origen_edit').prop('disabled', true).prop('required', false).selectpicker('val', '').selectpicker('refresh');

        $destinoDiv.slideDown();
        $('#institucion_destino_edit').prop('disabled', false).prop('required', true).selectpicker('refresh');

    } else {
        $origenDiv.slideUp();
        $('#institucion_origen_edit').prop('disabled', true).prop('required', false).selectpicker('val', '').selectpicker('refresh');

        $destinoDiv.slideUp();
        $('#institucion_destino_edit').prop('disabled', true).prop('required', false).selectpicker('val', '').selectpicker('refresh');
    }
}

// Conecta los listeners del MODAL DE CREAR
function initEventHandlers() {
    $('#filtroInstitucionOrigen').on('change', handleOrigenChange);
    $('#filtroInstitucionDestino').on('change', function () {
        actualizarOpcionesOpuestas($(this), $('#filtroInstitucionOrigen'));
    });

    // --- MODAL CREAR: Al abrir ---
    $('#modalNuevoTraslado').on('show.bs.modal', function () {
        $('#formNuevoTraslado')[0].reset();
        $('#student_id').val('');
        $('#nombre_estudiante').val('(Resultados de la búsqueda)');
        $('#formNuevoTraslado').removeClass('was-validated');

        $('#institucion_origen').closest('.col-md-6').hide();
        $('#institucion_destino').closest('.col-md-6').hide();
        $('#institucion_origen').prop('disabled', true).prop('required', false).selectpicker('val', '');
        $('#institucion_destino').prop('disabled', true).prop('required', false).selectpicker('val', '');
        $('#tipo').selectpicker('val', '');

        loadModalInstitutions();
    });

    // --- MODAL CREAR: Al cambiar tipo ---
    $('#tipo').on('change', handleTipoTrasladoChange);

    // --- MODAL CREAR: Al buscar DNI ---
    $('#btnBuscarDocumento').on('click', function () {
        const nroidenti = $('#nro_documento_busqueda').val();
        if (!nroidenti) {
            notificaciontoast("warning", "Ingrese un número de documento", "Advertencia");
            return;
        }
        const $btn = $(this);
        $btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i>');
        $.ajax({
            url: '/admin/estudiantes/buscar',
            type: 'POST',
            data: { _token: csrfToken, nroidenti: nroidenti },
            success: function (data) {
                $('#student_id').val(data.student_id);
                const nombreCompleto = `${data.nombres} ${data.apellido_pa} ${data.apellido_ma}`;
                $('#nombre_estudiante').val(nombreCompleto);
                notificaciontoast("success", "Estudiante encontrado", "Éxito");
            },
            error: function (xhr) {
                $('#student_id').val('');
                $('#nombre_estudiante').val('(Estudiante no encontrado)');
                let errorMsg = xhr.responseJSON ? xhr.responseJSON.error : "Error al buscar";
                notificaciontoast("error", errorMsg, "Error");
            },
            complete: function () {
                $btn.prop('disabled', false).html('<i class="fas fa-search"></i>');
            }
        });
    });

    // --- MODAL CREAR: Al guardar ---
    $('#formNuevoTraslado').on('submit', function (e) {
        e.preventDefault();
        if (!this.checkValidity()) {
            e.stopPropagation(); $(this).addClass('was-validated'); return;
        }
        if (!$('#student_id').val()) {
            notificaciontoast("error", "Debe buscar y seleccionar un estudiante válido", "Error");
            $(this).removeClass('was-validated'); return;
        }

        const $btnGuardar = $('#btnGuardarTraslado');
        $btnGuardar.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Guardando...');
        const formData = new FormData(this);

        $.ajax({
            url: '/admin/estudiantes/traslados/store',
            type: 'POST',
            data: formData,
            processData: false, contentType: false,
            headers: { 'X-CSRF-TOKEN': csrfToken },
            success: function (response) {
                notificaciontoast("success", response.success, "Éxito");
                $('#modalNuevoTraslado').modal('hide');
                loadDataTable();
            },
            error: function (xhr) {
                let errorMsg = "No se pudo guardar.";
                if (xhr.responseJSON && xhr.responseJSON.error) { errorMsg = xhr.responseJSON.error; }
                notificaciontoast("error", errorMsg, "Error");
            },
            complete: function () {
                $btnGuardar.prop('disabled', false).html('<i class="fas fa-save"></i> Guardar');
            }
        });
    });
}

// Conecta los NUEVOS listeners para Ver, Editar y Eliminar
function initNewModalHandlers() {

    $(document).on('click', '.btn-view-document', function () {
        const path = $(this).data('path');
        const filename = $(this).data('filename');

        // Codificamos la ruta para que maneje espacios y caracteres como 'N°'
        const encodedPath = encodeURI(path);
        const url = `/admin/estudiantes/traslados/ver/${encodedPath}`;

        // Reseteamos todos los contenedores
        $('#documentViewer').addClass('d-none'); // <-- Oculta el padre
        $('#pdfViewer, #imageViewer, #unsupportedAlert').addClass('d-none');

        // Seteamos la info
        $('#documentTitle').text(filename);
        $('#fileNameInfo').text(filename);
        $('#downloadLink').attr('href', url);

        const lowerFilename = filename.toLowerCase();

        if (lowerFilename.endsWith('.pdf')) {
            $('#documentViewer').removeClass('d-none');
            $('#pdfFrame').attr('src', url);
            $('#pdfViewer').removeClass('d-none');

        } else if (lowerFilename.endsWith('.jpg') || lowerFilename.endsWith('.jpeg') || lowerFilename.endsWith('.png')) {
            $('#documentViewer').removeClass('d-none');
            $('#imageFrame').attr('src', url);
            $('#imageViewer').removeClass('d-none');

        } else {
            $('#unsupportedMessage').text(`El archivo "${filename}" no se puede previsualizar.`);
            $('#unsupportedAlert').removeClass('d-none');
        }

        $('#modalViewDocument').modal('show');
    });

    $(document).on('click', '.btn-edit-traslado', function () {
        const id = $(this).data('id');

        $('#formEditarTraslado')[0].reset();
        $('#formEditarTraslado').removeClass('was-validated');
        $('#modalEditarTrasladoLabel').html('<i class="fas fa-spinner fa-spin"></i> Cargando datos...');
        $('#modalEditarTraslado').modal('show');

        loadModalInstitutions_Edit().then(() => {
            $.ajax({
                url: `/admin/estudiantes/traslados/show/${id}`,
                type: 'GET',
                dataType: 'json',
                success: function (data) {
                    $('#modalEditarTrasladoLabel').html('<i class="fas fa-pencil-alt"></i> Editar Traslado');

                    $('#traslado_id_edit').val(data.id);
                    $('#student_id_edit').val(data.student_id);
                    $('#nro_documento_edit').val(data.nroidenti);
                    $('#nombre_estudiante_edit').val(data.nombre_completo);
                    $('#fecha_edit').val(data.fecha);
                    $('#carrera_origen_edit').val(data.carrera_origen);
                    $('#carrera_destino_edit').val(data.carrera_destino);


                    $('#tipo_edit').selectpicker('val', data.tipo);

                    if (data.tipo === 'Ingreso') {
                        $('#institucion_origen_edit').selectpicker('val', data.institucion_origen);
                    } else if (data.tipo === 'Salida') {
                        $('#institucion_destino_edit').selectpicker('val', data.institucion_destino);
                    }

                    // Esto ejecutará handleTipoTrasladoChange_Edit() y ocultará el select automático
                    $('#tipo_edit').trigger('change');
                },
                error: function (xhr) {
                    notificaciontoast("error", "No se pudieron cargar los datos del traslado.", "Error");
                    $('#modalEditarTraslado').modal('hide');
                }
            });
        }).catch(() => {
            notificaciontoast("error", "Error al cargar instituciones. No se puede abrir el modal.", "Error");
            $('#modalEditarTraslado').modal('hide');
        });
    });

    $('#tipo_edit').on('change', handleTipoTrasladoChange_Edit);

    $('#formEditarTraslado').on('submit', function (e) {
        e.preventDefault();
        if (!this.checkValidity()) {
            e.stopPropagation();
            $(this).addClass('was-validated');
            return;
        }

        const $btnGuardar = $('#btnGuardarEdicion');
        $btnGuardar.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Actualizando...');
        const formData = new FormData(this);
        const trasladoId = $('#traslado_id_edit').val();

        $.ajax({
            url: `/admin/estudiantes/traslados/update/${trasladoId}`,
            type: 'POST',
            data: formData,
            processData: false, contentType: false,
            headers: { 'X-CSRF-TOKEN': csrfToken },
            success: function (response) {
                notificaciontoast("success", response.success, "Éxito");
                $('#modalEditarTraslado').modal('hide');
                loadDataTable();
            },
            error: function (xhr) {
                let errorMsg = "No se pudo actualizar el registro.";
                if (xhr.responseJSON && xhr.responseJSON.error) { errorMsg = xhr.responseJSON.error; }
                notificaciontoast("error", errorMsg, "Error");
            },
            complete: function () {
                $btnGuardar.prop('disabled', false).html('<i class="fas fa-save"></i> Actualizar');
            }
        });
    });
    $(document).on('click', '.btn-delete-traslado', function () {
        const id = $(this).data('id');
        // Llama a la función de borrado directamente
        destroyTraslado(id);
    });
}

// ==========================================================
// FUNCIÓN: Eliminar Traslado
// ==========================================================
function destroyTraslado(id) {
    $.ajax({
        url: `/admin/estudiantes/traslados/destroy/${id}`,
        type: 'DELETE',
        headers: { 'X-CSRF-TOKEN': csrfToken },
        dataType: 'json',
        success: function (response) {
            notificaciontoast("success", response.success || "Traslado eliminado", "Éxito");
            loadDataTable();
        },
        error: function (xhr) {
            let errorMsg = xhr.responseJSON ? xhr.responseJSON.error : "No se pudo eliminar el registro.";
            notificaciontoast("error", errorMsg, "Error");
        }
    });
}

// ==========================================================
// FUNCIONES AUXILIARES (Tu código original)
// ==========================================================

function actualizarOpcionesOpuestas($sourceSelect, $targetSelect) {
    const selectedValue = $sourceSelect.val();
    $targetSelect.find('option').prop('disabled', false);

    if (selectedValue && selectedValue !== "0") {
        $targetSelect.find(`option[value="${selectedValue}"]`).prop('disabled', true);
    }
    $targetSelect.selectpicker('refresh');
    aplicarEstiloTachado($targetSelect);
}

function handleOrigenChange() {
    const origenValue = $(this).val();
    const $destinoSelect = $('#filtroInstitucionDestino');

    if (origenValue && origenValue !== "0") {
        $destinoSelect.prop('disabled', false);
    } else {
        $destinoSelect.prop('disabled', true);
        $destinoSelect.val('0');
    }

    $destinoSelect.find('option').prop('disabled', false);
    if (origenValue && origenValue !== "0") {
        $destinoSelect.find(`option[value="${origenValue}"]`).prop('disabled', true);
    }

    $destinoSelect.selectpicker('refresh');
    aplicarEstiloTachado($destinoSelect);
}

function aplicarEstiloTachado($select) {
    setTimeout(function () {
        const $dropdownMenu = $select.parent().find('.dropdown-menu');
        $dropdownMenu.find('a').each(function () {
            const $link = $(this);
            const $li = $link.parent();
            if ($li.hasClass('disabled')) {
                $link.css({ 'text-decoration': 'line-through', 'opacity': '0.7' });
            } else {
                $link.css({ 'text-decoration': 'none', 'opacity': '1' });
            }
        });
    }, 100);
}