// ==========================================================
// 1. IMPORTS Y DEPENDENCIAS
// ==========================================================
import { notificaciontoast } from "../../common/method_toast.js";
import { confirmation_sweetAlert } from "../../common/method_sweetAlert.js";
import { getButtonsSchema, languageSchema } from "../../common/schemas.js";

// ==========================================================
// 2. VARIABLES GLOBALES
// ==========================================================

// ==========================================================
// 3. CONFIGURACIÓN CSRF
// ==========================================================
(function csrfBootstrap($) {
    function getCsrfToken() {
        const fromMeta = $('meta[name="csrf-token"]').attr('content');
        if (fromMeta) return fromMeta;

        const fromInput = $('input[name="_token"]').first().val();
        if (fromInput) return fromInput;

        const m = document.cookie.match(/(?:^|;\s*)XSRF-TOKEN=([^;]+)/);
        if (m && m[1]) {
            try { return decodeURIComponent(m[1]); } catch { return m[1]; }
        }
        return null;
    }

    $.ajaxPrefilter(function (options, originalOptions, jqXHR) {
        if (options.crossDomain) return;
        const token = getCsrfToken();
        if (token) {
            jqXHR.setRequestHeader('X-CSRF-TOKEN', token);
            jqXHR.setRequestHeader('X-XSRF-TOKEN', token);
        }
    });
})(jQuery);

// ==========================================================
// 4. ESTADO GLOBAL DE FILTROS
// ==========================================================
const Filters = {
    idprogram: "0",
    idplan: "0",
    idstudent: "0",
};

// ==========================================================
// 5. INICIALIZACIÓN PRINCIPAL
// ==========================================================
$(document).ready(function () {
    initEventHandlers();
    initDataTables();
    initTimeline();
});

// ==========================================================
// 6. HANDLERS DE EVENTOS
// ==========================================================
function initEventHandlers() {
    $(document).on("click", "#btnCreateModal", handleCreateModalClick);

    $('#formNuevaConvalidacion').on('submit', handleFormConvalidacionSubmit);
    $('#formAddCourse').on('submit', handleFormAddCourseSubmit);

    $('#datatableGeneral tbody').on('click', '.addCourseBtn', handleAddCourseClick);
    $('#datatableEquivalences tbody').on('click', '.delEquivBtn', handleDeleteEquivalence);
    $('#datatableGeneral tbody').on('click', '.viewDocBtn', handleViewDocument);
    $('#datatableGeneral tbody').on('click', '.deleteConvBtn', handleDeleteConvalidacion);

    $("#tlprogram").on("click", ".btnProgram", handleProgramClick);
    $("#txt-student").on("change", handleStudentChange);

    $('#modalViewDocument').on('hidden.bs.modal', resetViewers);

}

// ==========================================================
// 7. MODALES
// ==========================================================
function handleCreateModalClick() {
    if (Filters.idstudent === "0" || Filters.idstudent === 0) {
        notificaciontoast("warning", "Primero seleccione un estudiante", "Advertencia!");
        return;
    }
    showNuevaConvalidacionModal();
}

function showNuevaConvalidacionModal() {
    $('#student_id').val(Filters.idstudent);
    $('#formNuevaConvalidacion')[0].reset();
    $('#fecha_registro').val(new Date().toISOString().split('T')[0]);
    $('#modalNuevaConvalidacion').modal('show');
}

// ==========================================================
// 8. DATATABLES
// ==========================================================
let table;
let tableEquivalences;

function initDataTables() {
    table = $("#datatableGeneral").DataTable({
        createdRow: row => $(row).addClass("selected"),
        order: [],
        ajax: {
            headers: { "X-CSRF-TOKEN": $('input[name="_token"]').val() },
            type: "GET",
            url: "/admin/estudiantes/convalidaciones/listar",
            data: d => ({ ...d, ...Filters }),
            dataSrc: json => (Filters.idstudent === "0" ? [] : (Array.isArray(json) ? json : (json.data || []))),
        },
        searching: true,
        pageLength: 12,
        dom: "Bfrtip",
        select: false,
        lengthChange: false,
        responsive: true,
        language: languageSchema,
        buttons: getButtonsSchema("Convalidaciones externas"),
        columns: getConvalidacionesColumns(),
    });

    tableEquivalences = $('#datatableEquivalences').DataTable({
        ajax: {
            url: '/admin/estudiantes/convalidaciones/cursos/list/0',
            dataSrc: json => Array.isArray(json) ? json : (json.data || []),
            error: xhr => {
                const msg = xhr.responseJSON?.mensaje || "Error al cargar equivalencias";
                notificaciontoast("error", msg, "Error!");
            }
        },
        searching: true,
        paging: true,
        pageLength: 8,
        ordering: true,
        lengthChange: false,
        language: languageSchema,
        columns: getEquivalenciasColumns(),
    });
}

// ==========================================================
// 9. COLUMNAS DE TABLAS
// ==========================================================
function getConvalidacionesColumns() {
    return [
        {
            data: "institucion_name",
            className: "text-center",
            defaultContent: "-",
            render: data => data || '-',
        },
        {
            data: "tipo_nombre",
            className: "text-center",
            defaultContent: "N/A",
            render: data => data || 'N/A',
        },
        {
            data: "fecha_registro",
            className: "text-center",
            defaultContent: "-",
            render: data => data ? new Date(data).toLocaleDateString('es-ES') : '-',
        },
        {
            data: "documento_url",
            className: "text-center noExport",
            orderable: false,
            searchable: false,
            render: renderDocumentButton,
        },
        {
            data: "id",
            className: "text-center noExport",
            orderable: false,
            searchable: false,
            render: id => `
                    <button class="btn btn-sm btn-outline-success addCourseBtn"
                            data-id="${id}" title="Registrar equivalencia de cursos">
                        <i class="fas fa-plus-circle"></i> Equivalencia
                    </button>`,
        },
        {
            data: "id",
            className: "text-center noExport",
            orderable: false,
            searchable: false,
            render: id => `
                    <button class="btn btn-xs btn-outline-danger deleteConvBtn"
                            data-id="${id}" title="Eliminar convalidación">
                        <i class="fas fa-trash"></i>
                    </button>`,
        }
    ];
}

function getEquivalenciasColumns() {
    return [
        { data: 'codigo', className: 'text-center', defaultContent: '-' },
        { data: 'course_name', defaultContent: '-' },
        { data: 'creditos', className: 'text-center', defaultContent: '-' },
        { data: 'horas_por_semana', className: 'text-center', defaultContent: '-' },
        {
            data: 'promedio_final',
            className: 'text-center',
            defaultContent: '-',
            render: data => data ? parseFloat(data).toFixed(1) : '-',
        },
        {
            data: 'observaciones',
            defaultContent: '-',
            render: data => data || '-',
        },
        {
            data: 'id',
            className: 'text-center',
            orderable: false,
            render: id => `
                    <button class="btn btn-xs btn-outline-danger delEquivBtn" data-id="${id}">
                        <i class="fas fa-trash"></i>
                    </button>`,
        }
    ];
}

// ==========================================================
// 10. FORMULARIOS
// ==========================================================
function handleFormConvalidacionSubmit(e) {
    e.preventDefault();
    const form = this;
    if (!form.checkValidity()) {
        form.classList.add('was-validated');
        return;
    }

    const btn = $('#btnGuardarConvalidacion');
    btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Guardando...');

    const formData = new FormData(form);

    $.ajax({
        headers: { "X-CSRF-TOKEN": $('input[name="_token"]').val() },
        type: "POST",
        url: "/admin/estudiantes/convalidaciones/store",
        data: formData,
        processData: false,
        contentType: false,
        success: response => {
            if (response.status) {
                notificaciontoast("success", response.mensaje, "Éxito!");
                $('#modalNuevaConvalidacion').modal('hide');
                table.ajax.reload();
            } else {
                notificaciontoast("error", response.mensaje, "Error!");
            }
        },
        error: xhr => {
            const msg = xhr.responseJSON?.mensaje || xhr.responseJSON?.errors?.join(', ') || "Error al guardar";
            notificaciontoast("error", msg, "Error!");
        },
        complete: () => btn.prop('disabled', false).html('<i class="fas fa-save"></i> Guardar'),
    });
}

function handleFormAddCourseSubmit(e) {
    e.preventDefault();
    const form = this;
    if (!form.checkValidity()) {
        form.classList.add('was-validated');
        return;
    }

    const btn = $(form).find('button[type="submit"]');
    btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Guardando...');

    $.ajax({
        headers: { "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr('content') },
        type: "POST",
        url: "/admin/estudiantes/convalidaciones/cursos/store",
        data: $(form).serialize(),
        success: r => {
            notificaciontoast('success', r.mensaje || 'Guardado', 'Éxito');
            form.reset();
            $(form).removeClass('was-validated');
            $('#cbCursosPlan').selectpicker('refresh');
            tableEquivalences.ajax.reload(null, false);
        },
        error: xhr => {
            const msg = xhr.responseJSON?.mensaje || 'Error al guardar';
            notificaciontoast('error', msg, 'Error');
        },
        complete: () => btn.prop('disabled', false).html('Guardar'),
    });
}

// ==========================================================
// 11. EQUIVALENCIAS
// ==========================================================
function handleAddCourseClick() {
    const idConv = $(this).data('id');
    $('#convalidacion_externa_id').val(idConv);
    $('#convalidation_id').val(idConv);
    $('#formAddCourse')[0].reset();
    $('#formAddCourse').removeClass('was-validated');

    if (!Filters.idplan || Filters.idplan === '0') {
        notificaciontoast('warning', 'No se ha seleccionado un plan', 'Advertencia');
        return;
    }

    cargarCursosDelPlan(Filters.idplan);
    tableEquivalences.ajax.url(`/admin/estudiantes/convalidaciones/cursos/list/${idConv}`).load();
    $('#modalAddCourse').modal('show');
}

function handleDeleteEquivalence() {
    const id = $(this).data('id');
    confirmation_sweetAlert("¿Eliminar?", "Esta acción no se puede deshacer.", "warning").then(res => {
        if (!res.value) return;
        $.ajax({
            headers: { "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr('content') },
            type: "DELETE",
            url: `/admin/estudiantes/convalidaciones/cursos/destroy/${id}`,
            success: r => {
                if (r.status) {
                    notificaciontoast('info', r.mensaje || 'Eliminado', 'Éxito');
                    tableEquivalences.ajax.reload(null, false);
                } else {
                    notificaciontoast('error', r.mensaje || 'No se pudo eliminar', 'Error');
                }
            },
            error: () => notificaciontoast('error', 'Error al eliminar', 'Error'),
        });
    });
}

function handleDeleteConvalidacion() {
    const id = $(this).data('id');
    confirmation_sweetAlert("¿Eliminar?", "Se marcará como eliminada y no podrá recuperarse.", "warning").then(res => {
        if (!res.value) return;

        $.ajax({
            headers: { "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr('content') },
            type: "DELETE",
            url: `/admin/estudiantes/convalidaciones/destroy/${id}`,
            success: r => {
                if (r.status) {
                    notificaciontoast('info', r.mensaje || 'Eliminado', 'Éxito');
                    table.ajax.reload(null, false);
                } else {
                    notificaciontoast('error', r.mensaje || 'No se pudo eliminar', 'Error');
                }
            },
            error: xhr => {
                console.log('Error completo:', xhr);
                console.log('Response JSON:', xhr.responseJSON);
                const msg = xhr.responseJSON?.mensaje || 'Error al guardar';
                notificaciontoast('error', msg, 'Error');
            },
            complete: () => btn.prop('disabled', false).html('Guardar')
        });
    });
}

// ==========================================================
// 12. CARGAR CURSOS DEL PLAN
// ==========================================================
function cargarCursosDelPlan(idplan) {
    const $cb = $('#cbCursosPlan');
    $cb.empty().append('<option value="">-- Cargando --</option>').selectpicker('refresh');

    $.get(`/ajax_all/cursos/by_plan/${idplan}`)
        .done(data => {
            $cb.empty().append('<option value="">-- Seleccione --</option>');
            data.forEach(c => $cb.append(`<option value="${c.id}">${c.codigo_curso} - ${c.nombre}</option>`));
            $cb.selectpicker('refresh');
        })
        .fail(xhr => {
            const msj = xhr.responseJSON?.mensaje || `Error ${xhr.status} al cargar cursos del plan`;
            notificaciontoast('error', msj, 'Error');
            $cb.empty().append('<option value="">-- Error al cargar --</option>').selectpicker('refresh');
        });
}

// ==========================================================
// 13. VISOR DE DOCUMENTOS
// ==========================================================
function renderDocumentButton(data, type, row) {
    if (!data) return '-';
    const fileName = row.documento ? row.documento.split('/').pop() : 'Documento';
    const ext = fileName.split('.').pop().toLowerCase();
    let icon = 'fa-file';
    if (ext === 'pdf') icon = 'fa-file-pdf text-danger';
    else if (['doc', 'docx'].includes(ext)) icon = 'fa-file-word text-primary';
    else if (['jpg', 'jpeg', 'png'].includes(ext)) icon = 'fa-file-image text-success';

    return `
            <button type="button" class="btn btn-outline-primary btn-xs viewDocBtn"
                    data-url="${data}" data-filename="${fileName}">
                <i class="fas ${icon}"></i> Ver doc
            </button>`;
}

function handleViewDocument() {
    const url = $(this).data('url');
    const fileName = $(this).data('filename') || 'Documento';
    const ext = (fileName.split('.').pop() || '').toLowerCase();

    $('#documentTitle').text('Visualizar Documento');
    $('#fileNameInfo').text(fileName);
    $('#downloadLink').attr('href', url).attr('target', '_blank').attr('rel', 'noopener noreferrer');

    resetViewers();

    if (ext === 'pdf') {
        $('#documentViewer').removeClass('d-none');
        $('#pdfViewer').removeClass('d-none');
        $('#pdfFrame').attr('src', url);
    } else if (['jpg', 'jpeg', 'png', 'gif', 'webp'].includes(ext)) {
        $('#documentViewer').removeClass('d-none');
        $('#imageViewer').removeClass('d-none');
        $('#imageFrame').attr('src', url);
    } else if (['txt', 'log', 'csv'].includes(ext)) {
        $('#unsupportedAlert').removeClass('d-none');
        $('#unsupportedMessage').text('Este tipo de archivo se descarga directamente.');
    } else if (['doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx'].includes(ext)) {
        $('#unsupportedAlert').removeClass('d-none');
        $('#unsupportedMessage').text('Este tipo de archivo no puede visualizarse en el navegador. Puedes descargarlo.');
    } else {
        $('#unsupportedAlert').removeClass('d-none');
        $('#unsupportedMessage').text('Tipo de archivo no soportado para previsualización.');
    }

    $('#modalViewDocument').modal('show');
}

function resetViewers() {
    $('#unsupportedAlert').addClass('d-none');
    $('#documentViewer').addClass('d-none');
    $('#pdfViewer, #imageViewer, #textViewer').addClass('d-none');
    $('#pdfFrame').attr('src', '');
    $('#imageFrame').attr('src', '');
    $('#textContent').text('');
}

// ==========================================================
// 14. TIMELINE
// ==========================================================
function initTimeline() {
    tl_programs();
}

function handleProgramClick() {
    Filters.idstudent = "0";
    Filters.idplan = "0";
    cleanInfoAlumno();
    $(".divprogram").removeClass("success").addClass("primary");
    $(this).prev("div").removeClass("primary").addClass("success");
    Filters.idprogram = $(this).data("idprogram");
    cb_students_by_program(Filters.idprogram);
}

function handleStudentChange() {
    Filters.idstudent = $(this).val();
    const opt = $(this).find("option:selected");
    Filters.idplan = opt.data("plan_id") || "0";
    const apenom = `${opt.data("apellido_pa")} ${opt.data("apellido_ma")}, ${opt.data("nombres")}`;
    $("#txt-info-nombres").text(apenom);
    $("#txt-info-id").text(`${opt.data("nroidenti")}`);
    $("#txt-info-anho_ingreso").text(opt.data("anho_ingreso"));
    notificaciontoast("info", "Consultando a la base de datos!", "Mensaje!");
    table.ajax.reload();
}

// ==========================================================
// 15. ESTUDIANTES
// ==========================================================
function cleanInfoAlumno() {
    $("#txt-info-nombres").text("--");
    $("#txt-info-id").text("--");
    $("#txt-info-anho_ingreso").text("--");
    $("#txt-student").empty()
        .append('<option disabled selected value="">Buscar alumno</option>')
        .selectpicker("refresh");
}

function cb_students_by_program(idprogram) {
    const $cb = $("#txt-student");
    $cb.empty().append('<option disabled selected value="">Cargando alumnos…</option>').selectpicker("refresh");

    // Usamos la ruta que apunta a studentByProgram en tu controlador
    $.get(`/ajax_all/student/by_idprograms/${idprogram}`)
        .done(rows => {
            $cb.empty().append('<option disabled selected value="0">Buscar alumno</option>');
            if (!rows || !rows.length) {
                $cb.append('<option disabled value="">(Sin alumnos con traslados para este programa)</option>');
            } else {
                rows.forEach(el => {
                    const apenom = `${el.apellido_pa || ""} ${el.apellido_ma || ""}, ${el.nombres || ""}`;
                    $cb.append(`<option value="${el.idstudent}"
                            data-subtext="${el.nroidenti || ""}"
                            data-idstudent="${el.idstudent}"
                            data-plan_id="${el.plan_id || "0"}" 
                            data-apellido_pa="${el.apellido_pa || ""}"
                            data-apellido_ma="${el.apellido_ma || ""}"
                            data-nombres="${el.nombres || ""}"
                            data-tipoidenti="${el.tipoidenti || ""}"
                            data-nroidenti="${el.nroidenti || ""}"
                            data-celular="${el.celular || ""}"
                            data-anho_ingreso="${el.anho_ingreso || ""}"
                            data-estado="${el.estado || ""}">${apenom}</option>`);
                });
            }
            $cb.selectpicker("refresh");
        })
        .fail(() => {
            $cb.empty().append('<option disabled selected value="">Error cargando alumnos</option>').selectpicker("refresh");
        });
}

// ==========================================================
// 16. TIMELINE FUNCTIONS
// ==========================================================
function tl_programs() {
    $.get("/ajax_all/programs", response => {
        const rows = response?.data || response || [];
        const ul = document.getElementById("tlprogram");
        ul.innerHTML = "";
        rows.forEach(e => {
            ul.innerHTML += `
                    <li>
                        <div class="divprogram timeline-badge primary"></div>
                        <a class="btnProgram timeline-panel text-muted px-2 py-1 mr-4 ml-2" href="javascript:void(0);"
                            data-idprogram="${e.id}" data-codprograma="${e.codprograma}" data-nombre="${e.nombre}">
                            <h6 class="mb-0">
                                <strong class="text-info">${e.codprograma || ""}</strong><br>${e.nombre}
                            </h6>
                        </a>
                    </li>`;
        });
    });
}

