<?php

namespace App\Http\Middleware;

use App\Models\Module;
use App\Services\AuthService;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckModulePermission
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, $moduleShortName): Response
    {
        $user = Auth::User();
        // verificar si hay usuario
        if (!$user) {
            (new AuthService())->logout($request);
            return redirect()->route('login');
        }

        $loggedUserType = $request->session()->get('loggedUserType');

        $fail = fn(?string $msg = null) => $this->denyAccess($request, $user, $loggedUserType, $msg);

        $administratorData = $user->administradores;
        if (!$administratorData || count($administratorData) == 0) return $fail('Sin datos de administrador.');

        $administrator = $administratorData[0];

        // obtiene el rol del admin
        $role = $administrator->role;

        // obtiene id del modulo definido en el middleware
        $module = Module::where('short_name', $moduleShortName)->first();

        if (!$role || !$module) return $fail('Rol o módulo inválido.');

        $moduleId = $module->id;
        // obtiene permisos del cargo
        $permissions = $role->permisos()
            ->where('module_id', $moduleId)
            ->where('estado', 1) // valida acceso al modulo
            ->get();

        // si no tiene permisos
        if ($permissions->isEmpty()) return $fail('No tienes permiso para acceder a este módulo.');

        // si pasa todas las validaciones
        $request->attributes->set('module', $module); // setear modulo por si hay middleware de accion
        $response = $next($request);
        $response->headers->set('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
        $response->headers->set('Cache-Control', 'post-check=0, pre-check=0', false);
        $response->headers->set('Pragma', 'no-cache');
        $response->headers->set('Expires', '0');
        return  $response;
    }

    private function denyAccess(Request $request, $user, $loggedUserType, string $message)
    {

        // redireccionar según el rol del usuario
        if ($user->administradores->isNotEmpty() && $loggedUserType == 'admin') {
            $ruta = 'home.adm.dashboard';
        } else  if ($user->teachers->isNotEmpty() && $loggedUserType == 'teacher') {
            $ruta = 'home.docente.index';
        } else  if ($user->students->isNotEmpty() && $loggedUserType == 'student') {
            $ruta = 'home.student.index';
        } else {
            (new AuthService())->logout($request);
            $ruta = 'login';
        }

        if ($request->expectsJson()) {
            return response()->json([
                'redirect' => route($ruta),
                'status' => false,
                'mensaje' => $message ?? 'Acceso denegado.'
            ]);
        }

        return redirect()->route($ruta);
    }
}
