<?php

namespace App\Http\Middleware;

use App\Services\AuthService;
use App\Services\RouteService;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class CheckActionPermission
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, $actionName): Response
    {
        $user = Auth::user();
        // verificar si hay usuario
        if (!$user) {
            (new AuthService())->logout($request);
            return redirect()->route('login');
        }

        $loggedUserType = $request->session()->get('loggedUserType');

        $fail = fn(?string $msg = null) => $this->denyAccess($request, $user, $loggedUserType, $msg);

        // obtiene el modulo
        $module = $request->attributes->get('module');
        if (!$module) return $fail('Módulo inválido.');

        $administratorData = $user->administradores;
        if (!$administratorData || count($administratorData) == 0) return $fail('Sin datos de administrador.');

        $administrator = $administratorData[0];

        // obtiene el rol del admin
        $role = $administrator->role;
        if (!$role) return $fail('Rol inválido.');


        $moduleId = $module->id;
        $permissions = $role->permisos()
            ->where('module_id', $moduleId)
            ->where('estado', 1) // valida acceso al modulo
            ->where($actionName, 1) // valida acceso a la accion: 'create', 'read', 'update', 'delete'
            ->get();

        // si no tiene permisos
        if ($permissions->isEmpty()) return $fail('No tienes permiso para realizar esta acción sobre este módulo.');

        // si pasa todas las validaciones
        $response = $next($request);
        $response->headers->set('Cache-Control', 'no-store, no-cache, must-revalidate, max-age=0');
        $response->headers->set('Cache-Control', 'post-check=0, pre-check=0', false);
        $response->headers->set('Pragma', 'no-cache');
        $response->headers->set('Expires', '0');
        return  $response;
    }

    private function denyAccess(Request $request, $user, $loggedUserType, string $message)
    {
        // redireccionar según el rol del usuario
        if ($user->administradores->isNotEmpty() && $loggedUserType == 'admin') {
            $ruta = 'home.adm.dashboard';
        } else  if ($user->teachers->isNotEmpty() && $loggedUserType == 'teacher') {
            $ruta = 'home.docente.index';
        } else  if ($user->students->isNotEmpty() && $loggedUserType == 'student') {
            $ruta = 'home.student.index';
        } else {
            (new AuthService())->logout($request);
            $ruta = 'login';
        }

        if ($request->expectsJson()) {
            // return response()->json([
            //     'redirect' => route($route),
            //     'status' => false,
            //     'message' => $message ?? 'Acceso denegado.'
            // ], 403);
            return response()->json([
                'redirect' => route($ruta),
                'status' => false,
                'mensaje' => $message ?? 'Acceso denegado.'
            ]);
        }

        return redirect()->route($ruta);
    }
}
