<?php

namespace App\Http\Controllers\reportes;

use App\Http\Controllers\Controller;
use App\Models\Plan;
use App\Models\Student;
use App\Providers\AppServiceProvider;
use App\Services\QrGeneratorService;
use Barryvdh\DomPDF\Facade\Pdf as PDF;
use Endroid\QrCode\Builder\Builder;
use Endroid\QrCode\Encoding\Encoding;
use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\Writer\PngWriter;
use Endroid\QrCode\Writer\Result\GdResult;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class ReportesController extends Controller
{
    public function index1(Request $request)
    {
        return view('admin.notas.gestionar_notas.index');
    }

    public function index2(Request $request)
    {
        return view('admin.notas.reporte_notas.index');
    }

    public function list(Request $request)
    {
        $idsemester = $request->input('idsemester');
        $idperiod = $request->input('idperiod');

        if ($request->ajax()) {
            // query
            $data = DB::table('asignaturas as su')
                ->join('semestres as se', 'su.semester_id', '=', 'se.id')
                ->join('cursos as co', 'su.course_id', '=', 'co.id')
                ->join('periodos as pe', 'co.period_id', '=', 'pe.id')
                ->leftJoin('docentes as te', 'su.teacher_id', '=', 'te.id')
                ->leftJoin('usuarios as us', 'te.user_id', '=', 'us.id')
                ->select(
                    'su.id as idsubject',
                    'su.seccion',
                    'su.turno',
                    'su.tipo as condicion',
                    'su.nota_minima',
                    'su.teacher_id',
                    'co.codcurso',
                    'co.nombre as nomcurso',
                    'co.creditos',
                    'co.horas',
                    'co.tipo',
                    'se.id as idsemester',
                    'co.id as idcourse',
                    'pe.id as idperiod',
                    'us.apellido_pa',
                    'us.apellido_ma'
                )
                ->where('se.id', $idsemester)
                ->where('pe.id', $idperiod)
                ->orderBy('co.tipo', 'desc')
                ->orderBy('co.nombre', 'asc')
                ->get();

            // datatable
            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return response()->json(['status' => false, 'mensaje' => 'Error: no se pueden cargar los archivos']);
    }

    public function nota_asignatura($idsemester, $idsubject)
    {
        $infos = DB::table('asignaturas')
            ->join('cursos', 'cursos.id', '=', 'asignaturas.course_id')
            ->join('periodos', 'periodos.id', '=', 'cursos.period_id')
            ->join('planes', 'planes.id', '=', 'periodos.plan_id')
            ->join('programas', 'programas.id', '=', 'planes.program_id')
            ->crossJoin('semestres')
            ->select(
                'asignaturas.id as idsubject',
                'cursos.codcurso as codcurso',
                'cursos.nombre as curso',
                'cursos.tipo as tipo',
                'asignaturas.tipo as condicion',
                'periodos.numero as periodo',
                DB::raw("CONCAT('Plan ', planes.nombre, ' (', planes.tipo, ')') as plan"),
                'programas.nombre as programa',
                'programas.nivel_formativo',
                DB::raw("CONCAT(semestres.anho, '-', semestres.numero) as semestre")
            )
            ->where('asignaturas.id', $idsubject)
            ->where('semestres.id', $idsemester)
            ->get();

        $usuarios = DB::table('asignaturas')
            ->join('matriculas_asignaturas', 'asignaturas.id', '=', 'matriculas_asignaturas.subject_id')
            ->join('estudiantes', 'estudiantes.id', '=', 'matriculas_asignaturas.student_id')
            ->join('usuarios', 'usuarios.id', '=', 'estudiantes.user_id')
            ->join('semestres', 'semestres.id', '=', 'asignaturas.semester_id')
            ->join('cursos', 'cursos.id', '=', 'asignaturas.course_id')
            ->join('tipos_identificaciones AS it', 'it.id', '=', 'usuarios.identificationtype_id')
            ->select(
                'cursos.nombre as curso',
                'it.id AS ididentificationtype',
                'it.tipo AS tipoidenti',
                'usuarios.nroidenti',
                DB::raw('CONCAT(usuarios.nombres, " ", usuarios.apellido_pa, " ", usuarios.apellido_ma) as estudiante'),
                'matriculas_asignaturas.nota'
            )
            ->where('asignaturas.id', $idsubject)
            ->get();

        $pdf = PDF::loadView('reportes.nota_asignatura', [
            'institutions' => AppServiceProvider::$institutions,
            'infos' => $infos,
            'usuarios' => $usuarios,
        ]);

        // $pdf->setPaper('a4', 'landscape');
        // return $pdf->download('reporte_'.$cadenaAleatoria.'.pdf');
        return $pdf->stream();
    }

    public function nota_periodo($idsemester, $idperiod)
    {
        $infos = DB::table('periodos')
            ->join('planes', 'planes.id', '=', 'periodos.plan_id')
            ->join('programas', 'programas.id', '=', 'planes.program_id')
            ->crossJoin('semestres')
            ->select(
                'periodos.numero as periodo',
                DB::raw("CONCAT('Plan ', planes.nombre, ' (', planes.tipo, ')') as plan"),
                'programas.nombre as programa',
                'programas.nivel_formativo',
                DB::raw("CONCAT(semestres.anho, '-', semestres.numero) as semestre")
            )
            ->where('periodos.id', '=', $idperiod)
            ->where('semestres.id', '=', $idsemester)
            ->get();

        // Obtener la lista de cursos
        $querycursos = DB::table('asignaturas')
            ->join('semestres', 'semestres.id', '=', 'asignaturas.semester_id')
            ->join('cursos', 'cursos.id', '=', 'asignaturas.course_id')
            ->join('periodos', 'periodos.id', '=', 'cursos.period_id')
            ->select(
                'cursos.nombre',
                'cursos.tipo as tipo',
                'asignaturas.tipo as condicion'
            )
            ->where('semestres.id', '=', $idsemester)
            ->where('periodos.id', '=', $idperiod);

        // Construir la consulta dinámica
        $query = DB::table('asignaturas')
            ->join('matriculas_asignaturas', 'asignaturas.id', '=', 'matriculas_asignaturas.subject_id')
            ->join('estudiantes', 'estudiantes.id', '=', 'matriculas_asignaturas.student_id')
            ->join('usuarios', 'usuarios.id', '=', 'estudiantes.user_id')
            ->join('semestres', 'semestres.id', '=', 'asignaturas.semester_id')
            ->join('cursos', 'cursos.id', '=', 'asignaturas.course_id')
            ->join('periodos', 'periodos.id', '=', 'cursos.period_id')
            ->select(
                'usuarios.nroidenti',
                DB::raw('CONCAT(usuarios.nombres, " ", usuarios.apellido_pa, " ", usuarios.apellido_ma) AS estudiante')
            );

        $cursos = $querycursos->pluck('nombre');
        foreach ($cursos as $curso) {
            $nombre_columna = str_replace(' ', '_', $curso);
            $nombre_columna = str_replace('-', ' ', $nombre_columna);
            $query->addSelect(DB::raw('MAX(CASE WHEN cursos.nombre = "'.$curso.'" THEN COALESCE(asignaturas.nota_minima, "NM") END) AS '.$nombre_columna));
        }

        $query->where('periodos.id', '=', $idperiod)
            ->where('semestres.id', '=', $idsemester)
            ->groupBy('usuarios.nroidenti', 'estudiante');

        // Ejecutar la consulta
        $usuarios = $query->get();
        $nomcursos = $querycursos->orderBy('tipo', 'desc')->get();

        $pdf = PDF::loadView('reportes.nota_periodo', [
            'institutions' => AppServiceProvider::$institutions,
            'infos' => $infos,
            'usuarios' => $usuarios,
            'nomcursos' => $nomcursos,
        ]);

        return $pdf->stream();
    }

    public function curso_semestre_alumno($idsemester, $idstudent)
    {
        $cursos = DB::table('matriculas_asignaturas')
            ->join('asignaturas', 'asignaturas.id', '=', 'matriculas_asignaturas.subject_id')
            ->join('cursos', 'cursos.id', '=', 'asignaturas.course_id')
            ->join('periodos', 'periodos.id', '=', 'cursos.period_id')
            ->join('semestres', 'semestres.id', '=', 'asignaturas.semester_id')
            ->join('estudiantes', 'estudiantes.id', '=', 'matriculas_asignaturas.student_id')
            ->select(
                'cursos.codcurso',
                'cursos.nombre AS curso',
                'cursos.tipo',
                'cursos.creditos',
                'periodos.numero AS periodo',
                DB::raw('CONCAT(semestres.anho, "-", semestres.numero) AS semestre'),
                DB::raw('COALESCE(matriculas_asignaturas.nota, "NM") AS nota'),
            )
            ->where('matriculas_asignaturas.student_id', $idstudent)
            ->where('semestres.id', $idsemester)
            ->orderByDesc('semestre')
            ->orderBy('periodo')
            ->orderBy('curso')
            ->get();

        $estudiantes = DB::table('estudiantes')
            ->join('usuarios', 'usuarios.id', '=', 'estudiantes.user_id')
            ->join('planes', 'planes.id', '=', 'estudiantes.plan_id')
            ->join('programas', 'programas.id', '=', 'planes.program_id')
            ->crossJoin('semestres')
            ->select(
                'usuarios.nroidenti',
                DB::raw('CONCAT(usuarios.apellido_pa, " ", usuarios.apellido_ma, ", ", usuarios.nombres) AS estudiante'),
                'programas.nombre AS programa',
                'programas.nivel_formativo',
                DB::raw('CONCAT("Plan ", planes.nombre, " (", planes.tipo, ")") AS plan'),
                DB::raw('CONCAT(semestres.anho, "-", semestres.numero) AS semestre')
            )
            ->where('estudiantes.id', $idstudent)
            ->where('semestres.id', $idsemester)
            ->get();

        $pdf = PDF::loadView('reportes.curso_semestre_alumno', [
            'institutions' => AppServiceProvider::$institutions,
            'estudiantes' => $estudiantes,
            'cursos' => $cursos,
        ]);

        // $pdf->setPaper('a4', 'landscape');
        // return $pdf->download('reporte_'.$cadenaAleatoria.'.pdf');
        return $pdf->stream();
    }

    public function curso_total_alumno($idsemester, $idstudent)
    {
        $cursos = DB::table('matriculas_asignaturas')
            ->join('asignaturas', 'asignaturas.id', '=', 'matriculas_asignaturas.subject_id')
            ->join('cursos', 'cursos.id', '=', 'asignaturas.course_id')
            ->join('periodos', 'periodos.id', '=', 'cursos.period_id')
            ->join('semestres', 'semestres.id', '=', 'asignaturas.semester_id')
            ->join('estudiantes', 'estudiantes.id', '=', 'matriculas_asignaturas.student_id')
            ->select(
                'cursos.codcurso',
                'cursos.nombre AS curso',
                'cursos.tipo',
                'cursos.creditos',
                'periodos.numero AS periodo',
                DB::raw('CONCAT(semestres.anho, "-", semestres.numero) AS semestre'),
                DB::raw('COALESCE(matriculas_asignaturas.nota, "NM") AS nota'),
            )
            ->where('matriculas_asignaturas.student_id', $idstudent)
            ->orderByDesc('semestre')
            ->orderBy('curso')
            ->get();

        $estudiantes = DB::table('estudiantes')
            ->join('usuarios', 'usuarios.id', '=', 'estudiantes.user_id')
            ->join('planes', 'planes.id', '=', 'estudiantes.plan_id')
            ->join('programas', 'programas.id', '=', 'planes.program_id')
            ->crossJoin('semestres')
            ->select(
                'usuarios.nroidenti',
                DB::raw('CONCAT(usuarios.apellido_pa, " ", usuarios.apellido_ma, ", ", usuarios.nombres) AS estudiante'),
                'programas.nombre AS programa',
                'programas.nivel_formativo',
                DB::raw('CONCAT("Plan ", planes.nombre, " (", planes.tipo, ")") AS plan'),
                DB::raw('CONCAT(semestres.anho, "-", semestres.numero) AS semestre')
            )
            ->where('estudiantes.id', $idstudent)
            ->where('semestres.id', $idsemester)
            ->get();

        $pdf = PDF::loadView('reportes.curso_total_alumno', [
            'institutions' => AppServiceProvider::$institutions,
            'estudiantes' => $estudiantes,
            'cursos' => $cursos,
        ]);

        return $pdf->stream();
    }

    public function acta_oficial_notas($idsubject)
    {
        $infos = DB::table('asignaturas as a')
            ->select([
                DB::raw('CONCAT(se.anho, "-", se.numero) as semestre'),
                'pr.codprograma',
                'pr.nombre as programa',
                'pr.nivel_formativo as nivel_formativo',
                'pl.nombre as plan',
                'c.codcurso',
                'c.nombre as curso',
                DB::raw('CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) as creditos_horas'),
                'pe.numero as periodo',
                'a.tipo as condicion',
                'a.seccion',
                'a.turno',
            ])
            ->join('cursos as c', 'c.id', '=', 'a.course_id')
            ->join('periodos as pe', 'pe.id', '=', 'c.period_id')
            ->join('planes as pl', 'pl.id', '=', 'pe.plan_id')
            ->join('programas as pr', 'pr.id', '=', 'pl.program_id')
            ->join('semestres as se', 'se.id', '=', 'a.semester_id')
            ->where('a.id', '=', $idsubject)
            ->orderBy('c.nombre')
            ->get();

        $indicadores2 = DB::table('indicadores as i')
            ->join('asignaturas as a', 'a.id', '=', 'i.subject_id')
            ->where('a.id', $idsubject)
            ->pluck('i.nombre')
            ->toArray();

        // Obteniendo los indicadores
        $indicadores = DB::table('indicadores as i')
            ->join('asignaturas as a', 'a.id', '=', 'i.subject_id')
            ->where('a.id', $idsubject)
            ->pluck('i.id')
            ->toArray();

        $numIndicadores = count($indicadores);

        // Subconsulta gradesQuery
        $gradesQuery = DB::table('estudiantes as e')
            ->join('usuarios as u', 'e.user_id', '=', 'u.id')
            ->join('tipos_identificaciones as ti', 'u.identificationtype_id', '=', 'ti.id')
            ->join('matriculas_asignaturas as ma', 'e.id', '=', 'ma.student_id')
            ->join('asignaturas as a', 'ma.subject_id', '=', 'a.id')
            ->leftJoin('indicadores as i', 'i.subject_id', '=', 'a.id')
            ->leftJoin('actividades as ac', 'ac.indicator_id', '=', 'i.id')
            ->leftJoin('notas as n', 'n.activity_id', '=', 'ac.id')
            ->leftJoin('notas_estudiantes as ne', function ($join) {
                $join->on('ne.grade_id', '=', 'n.id')
                    ->on('ne.subjectenrollment_id', '=', 'ma.id');
            })
            ->where('a.id', $idsubject)
            ->groupBy([
                'a.id',
                'e.id',
                'ti.tipo',
                'u.nroidenti',
                'u.apellido_pa',
                'u.apellido_ma',
                'u.nombres',
                'i.id',
                'ac.id',
                'n.id',
                'ne.id',
                'n.nombre',
                'n.porcentaje',
            ])
            ->selectRaw('
                a.id AS idsubject,
                e.id AS idstudent,
                CONCAT(u.nroidenti) AS identificacion,
                CONCAT(u.apellido_pa, " ", u.apellido_ma, ", ", u.nombres) AS estudiante,
                i.id AS idindicator,
                ac.id AS idactivity,
                COALESCE(n.id, 0) AS idgrade,
                COALESCE(n.nombre, "-") AS tipo_nota,
                COALESCE(n.porcentaje, 0) AS porcentaje,
                COALESCE(ne.id, 0) AS idstudentgrade,
                COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 2), 0) AS nota
            ');

        // Subconsulta activityQuery
        $activityQuery = DB::table(DB::raw("({$gradesQuery->toSql()}) as gradesQuery"))
            ->mergeBindings($gradesQuery)
            ->groupBy([
                'idsubject',
                'idstudent',
                'identificacion',
                'estudiante',
                'idindicator',
                'idactivity',
            ])
            ->selectRaw('
                idsubject,
                idstudent,
                identificacion,
                estudiante,
                idindicator,
                idactivity,
                COALESCE(ROUND(SUM(nota * porcentaje), 2), 0) AS promedio_actividades
            ');

        // Subconsulta indicatorQuery
        $indicatorQuery = DB::table(DB::raw("({$activityQuery->toSql()}) as activityQuery"))
            ->mergeBindings($activityQuery)
            ->groupBy([
                'idsubject',
                'idstudent',
                'identificacion',
                'estudiante',
                'idindicator',
            ])
            ->selectRaw('
                idsubject,
                idstudent,
                identificacion,
                estudiante,
                idindicator,
                COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 2), 0) AS promedio_indicadores
            ');

        // Construyendo la parte dinámica de la consulta final
        $selectPromedios = [];
        foreach ($indicadores as $id) {
            $selectPromedios[] = "ROUND(MAX(CASE WHEN idindicator = $id THEN promedio_indicadores ELSE NULL END), 2) AS promedio_indicador_$id";
        }

        $selectPromedios = implode(', ', $selectPromedios);

        // Consulta final
        $estudiantes = DB::table(DB::raw("({$indicatorQuery->toSql()}) as indicatorQuery"))
            ->mergeBindings($indicatorQuery)
            ->groupBy(['idsubject', 'idstudent', 'identificacion', 'estudiante'])
            ->selectRaw("
                idsubject,
                idstudent,
                identificacion,
                estudiante,
                $selectPromedios,
                ROUND(
                    (
                        ".implode(' + ', array_map(function ($id) {
                return "COALESCE(MAX(CASE WHEN idindicator = $id THEN ROUND(promedio_indicadores, 2) ELSE NULL END), 0)";
            }, $indicadores))."
                    ) / $numIndicadores
                , 2) AS promedio_final
            ")
            ->get();

        $pdf = PDF::loadView('reportes.acta_oficial_notas', [
            'institutions' => AppServiceProvider::$institutions,
            'infos' => $infos,
            'indicadores' => $indicadores,
            'indicadores2' => $indicadores2,
            'estudiantes' => $estudiantes,
        ]);

        return $pdf->stream();
    }

    public function acta_auxiliar_notas($idsubject)
    {
        $infos = DB::table('asignaturas as a')
            ->select([
                DB::raw('CONCAT(se.anho, "-", se.numero) as semestre'),
                'pr.codprograma',
                'pr.nombre as programa',
                'pr.nivel_formativo as nivel_formativo',
                'pl.nombre as plan',
                'c.codcurso',
                'c.nombre as curso',
                DB::raw('CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) as creditos_horas'),
                'pe.numero as periodo',
                'a.tipo as condicion',
                'a.seccion',
                'a.turno',
            ])
            ->join('cursos as c', 'c.id', '=', 'a.course_id')
            ->join('periodos as pe', 'pe.id', '=', 'c.period_id')
            ->join('planes as pl', 'pl.id', '=', 'pe.plan_id')
            ->join('programas as pr', 'pr.id', '=', 'pl.program_id')
            ->join('semestres as se', 'se.id', '=', 'a.semester_id')
            ->where('a.id', '=', $idsubject)
            ->orderBy('c.nombre')
            ->get();

        $notas = DB::table('notas as n')
            ->join('actividades as a', 'a.id', '=', 'n.activity_id')
            ->join('indicadores as i', 'i.id', '=', 'a.indicator_id')
            ->where('i.subject_id', $idsubject)
            ->select(
                'i.id as indicador_id',
                'i.nombre as indicador',
                'a.id as actividad_id',
                'a.nombre as actividad',
                'n.nombre as tiponota',
                DB::raw('CONCAT(ROUND(n.porcentaje*100,0), "%") as porcentaje'),
                'n.id as nota_id',
                'n.nombre as nota'
            )
            ->groupBy('i.id', 'a.id', 'n.id', 'i.nombre', 'a.nombre', 'n.nombre', 'n.porcentaje', 'n.nombre')
            ->orderBy('i.id')
            ->orderBy('a.nombre', 'asc')
            ->get();

        $queryNotas = '
            WITH ranked_data AS (
                SELECT
                    i.id AS idindicator,
                    i.nombre AS indicador,
                    a.id AS idactivity,
                    a.nombre AS actividad,
                    COUNT(n.id) AS contador,
                    ROW_NUMBER() OVER (PARTITION BY i.id ORDER BY a.id DESC) AS rn
                FROM notas AS n
                JOIN actividades AS a ON a.id = n.activity_id
                JOIN indicadores AS i ON i.id = a.indicator_id
                WHERE i.subject_id = :subjectId
                GROUP BY i.id, a.nombre, a.id, i.nombre
            )

            SELECT
                idindicator,
                indicador,
                idactivity,
                actividad,
                contador,
                CASE
                    WHEN rn = 1 THEN 1
                    ELSE 0
                END AS estado
            FROM ranked_data
            ORDER BY idindicator, actividad ASC;
            ';

        $actividad_notas = DB::select($queryNotas, ['subjectId' => $idsubject]);

        $query = '
            WITH NotasPorTipo AS (
                SELECT
                    e.id AS idstudent,
                    CONCAT(u.nroidenti) AS identificacion,
                    CONCAT(u.apellido_pa, " ", u.apellido_ma, ", ", u.nombres) AS estudiante,
                    i.id AS indicador_id,
                    ac.id AS actividad_id,
                    i.nombre AS indicador,
                    ac.nombre AS actividad,
                    n.nombre AS nota,
                    n.porcentaje AS porcentaje,
                    n.id AS nota_id,
                    n.nombre AS nombre_nota,
                    COALESCE(ROUND(ne.nota, 1), 0) AS nota_estudiante
                FROM estudiantes AS e
                INNER JOIN usuarios AS u ON u.id = e.user_id
                INNER JOIN matriculas_asignaturas AS ma ON e.id = ma.student_id
                INNER JOIN asignaturas AS a ON ma.subject_id = a.id
                INNER JOIN indicadores AS i ON a.id = i.subject_id
                INNER JOIN actividades AS ac ON i.id = ac.indicator_id
                INNER JOIN notas AS n ON ac.id = n.activity_id
                LEFT JOIN notas_estudiantes AS ne ON ne.grade_id = n.id AND ne.subjectenrollment_id = ma.id
                WHERE a.id = :subjectId
                GROUP BY e.id, u.nroidenti, u.apellido_pa, u.apellido_ma, u.nombres, i.id, ac.id, n.id, i.nombre, ac.nombre, n.nombre, n.porcentaje, ne.nota
            ),

            PromedioPorActividad AS (
                SELECT
                    idstudent,
                    identificacion,
                    estudiante,
                    indicador_id,
                    indicador,
                    actividad_id,
                    actividad,
                    ROUND(SUM(nota_estudiante * porcentaje) / SUM(porcentaje), 2) AS prom_actividad
                FROM NotasPorTipo
                GROUP BY idstudent, identificacion, estudiante, indicador_id, indicador, actividad_id, actividad
            ),

            PromedioPorIndicador AS (
                SELECT
                    idstudent,
                    identificacion,
                    estudiante,
                    indicador_id,
                    indicador,
                    ROUND(SUM(prom_actividad) / COUNT(actividad_id), 2) AS prom_indicador
                FROM PromedioPorActividad
                GROUP BY idstudent, identificacion, estudiante, indicador_id, indicador
            ),

            PromedioFinal AS (
                SELECT
                    idstudent,
                    identificacion,
                    estudiante,
                    ROUND(SUM(prom_indicador) / COUNT(indicador_id), 2) AS prom_final
                FROM PromedioPorIndicador
                GROUP BY idstudent, identificacion, estudiante
            ),

            RankedData AS (
                SELECT
                    npt.idstudent,
                    npt.identificacion,
                    npt.estudiante,
                    npt.indicador_id AS idindicador,
                    npt.indicador,
                    npt.actividad_id AS idactividad,
                    npt.actividad,
                    npt.nota,
                    npt.porcentaje,
                    npt.nota_id,
                    npt.nombre_nota,
                    npt.nota_estudiante,
                    ROUND(ppa.prom_actividad, 2) AS prom_actividad,
                    ROUND(ppi.prom_indicador, 2) AS prom_indicador,
                    ROUND(pf.prom_final, 2) AS prom_final,
                    ROW_NUMBER() OVER (PARTITION BY npt.idstudent, npt.actividad_id ORDER BY npt.nota_id DESC) AS rn_actividad,
                    ROW_NUMBER() OVER (PARTITION BY npt.idstudent, npt.indicador_id ORDER BY npt.actividad_id DESC, npt.nota_id DESC) AS rn_indicador,
                    ROW_NUMBER() OVER (PARTITION BY npt.idstudent ORDER BY npt.indicador DESC , npt.actividad_id DESC, npt.nota_id DESC) AS rn_final
                FROM NotasPorTipo AS npt
                JOIN PromedioPorActividad AS ppa
                    ON npt.idstudent = ppa.idstudent
                    AND npt.indicador_id = ppa.indicador_id
                    AND npt.actividad_id = ppa.actividad_id
                JOIN PromedioPorIndicador AS ppi
                    ON npt.idstudent = ppi.idstudent
                    AND npt.indicador_id = ppi.indicador_id
                JOIN PromedioFinal AS pf
                    ON npt.idstudent = pf.idstudent
                ORDER BY npt.estudiante, npt.indicador, npt.actividad, npt.nota_id
            )

            SELECT
                idstudent,
                identificacion,
                estudiante,
                idindicador,
                indicador,
                idactividad,
                actividad,
                nota,
                porcentaje,
                nombre_nota,
                ROUND(nota_estudiante, 1) AS nota_estudiante,
                ROUND(prom_actividad, 1) AS prom_actividad,
                ROUND(prom_indicador, 1) AS prom_indicador,
                ROUND(prom_final + 0.001, 0) AS prom_final,
                CASE
                    WHEN rn_actividad = 1 THEN 1
                    ELSE 0
                END AS estado_actividad,
                CASE
                    WHEN rn_indicador = 1 THEN 1
                    ELSE 0
                END AS estado_indicador,
                CASE
                    WHEN rn_final = 1 THEN 1
                    ELSE 0
                END AS estado_promfinal
            FROM RankedData
                ';

        $estudiantes = DB::select($query, ['subjectId' => $idsubject]);

        $currentUrl = URL::current();
        $link = $currentUrl;
        $qrCode = (new QrGeneratorService())->get_qr_by_link($link);

        $pdf = PDF::loadView('reportes.acta_auxiliar_notas', [
            'institutions' => AppServiceProvider::$institutions,
            'infos' => $infos,
            'estudiantes' => $estudiantes,
            'actividad_notas' => $actividad_notas,
            'notas' => $notas,
            'qrCode' => $qrCode,
        ]);

        $pdf->setPaper('a4', 'landscape');

        return $pdf->stream();
    }

    public function promedio_asignaturas($idstudent, $idsemester)
    {
        $infos = DB::table('estudiantes AS e')
            ->select(
                'pr.codprograma AS codprograma',
                'pr.nombre AS programa',
                'pr.nivel_formativo AS nivel_formativo',
                'pl.nombre AS plan',
                'pl.tipo AS tipoplan',
                'pl.modalidad AS modalidad',
                'pl.enfoque AS enfoque',
                DB::raw('CONCAT(se.anho, "-", se.numero) AS semestre'),
                'se.tipo AS tiposemestre',
                'e.id AS idstudent',
                DB::raw('CONCAT(ti.tipo, " - ", u.nroidenti) AS identificacion'),
                DB::raw('CONCAT(u.apellido_pa, " ", u.apellido_ma, ", ", u.nombres) AS estudiante')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->join('matriculas_semestres AS ms', 'ms.student_id', '=', 'e.id')
            ->join('semestres AS se', 'se.id', '=', 'ms.semester_id')
            ->where('e.id', '=', $idstudent)
            ->where('se.id', '=', $idsemester)
            ->get();

        // Subconsulta para ver cada nota
        $gradesQuery = DB::table('matriculas_asignaturas AS ma')
            ->selectRaw('pl.nombre AS plan,
            c.codcurso AS codcurso,
            c.nombre AS curso,
            CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) AS creditos_horas,
            pe.numero AS periodo,
            a.tipo AS condicion,
            a.seccion AS seccion,
            a.turno AS turno,
            i.id AS idindicator,
            ac.id AS idactivity,
            n.id AS idgrade,
            ne.id AS idstudentgrade,
            n.nombre AS tipo_nota,
            n.porcentaje AS porcentaje,
            COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 1), 0) AS nota')
            ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
            ->join('cursos AS c', 'c.id', '=', 'a.course_id')
            ->join('periodos AS pe', 'pe.id', '=', 'c.period_id')
            ->join('planes AS pl', 'pl.id', '=', 'pe.plan_id')
            ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
            ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
            ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
            ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
            ->leftJoin('notas_estudiantes AS ne', function ($join) {
                $join->on('ne.grade_id', '=', 'n.id')
                    ->on('ne.subjectenrollment_id', '=', 'ma.id');
            })
            ->where('ma.student_id', '=', $idstudent)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'idindicator', 'idactivity', 'idgrade', 'idstudentgrade', 'tipo_nota', 'porcentaje');

        // Subconsulta para promedios por actividad
        $activityQuery = DB::table(DB::raw('('.$gradesQuery->toSql().') AS subconsulta_2'))
            ->mergeBindings($gradesQuery)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'idindicator', 'idactivity')
            ->selectRaw('plan,
            codcurso,
            curso,
            creditos_horas,
            periodo,
            condicion,
            seccion,
            turno,
            idindicator,
            idactivity,
            COALESCE(ROUND(SUM(nota * porcentaje), 1), 0) AS promedio_actividades');

        // Subconsulta para promedios por indicador
        $indicatorQuery = DB::table(DB::raw('('.$activityQuery->toSql().') AS subconsulta_3'))
            ->mergeBindings($activityQuery)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'idindicator')
            ->selectRaw('plan,
            codcurso,
            curso,
            creditos_horas,
            periodo,
            condicion,
            seccion,
            turno,
            idindicator,
            COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 1), 0) AS promedio_indicadores');

        // Consulta final
        $asignaturas = DB::table(DB::raw('('.$indicatorQuery->toSql().') AS subconsulta_4'))
            ->mergeBindings($indicatorQuery)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno')
            ->selectRaw('plan,
            codcurso,
            curso,
            creditos_horas,
            periodo,
            condicion,
            seccion,
            turno,
            COALESCE(ROUND(SUM(promedio_indicadores) / COUNT(idindicator), 1), 0) AS promedio_final')
            ->get();

        $pdf = PDF::loadView('reportes.promedios_asignaturas', [
            // 'institutions' => $this->institutions,
            'institutions' => AppServiceProvider::$institutions,
            'infos' => $infos,
            'asignaturas' => $asignaturas,
        ]);

        return $pdf->stream();
    }

    public function promedio_asignaturas2($idsemester)
    {
        $idstudent = Auth::User()->students[0]->id;

        $infos = DB::table('estudiantes AS e')
            ->select(
                'pr.codprograma AS codprograma',
                'pr.nombre AS programa',
                'pr.nivel_formativo AS nivel_formativo',
                'pl.nombre AS plan',
                'pl.tipo AS tipoplan',
                'pl.modalidad AS modalidad',
                'pl.enfoque AS enfoque',
                DB::raw('CONCAT(se.anho, "-", se.numero) AS semestre'),
                'se.tipo AS tiposemestre',
                'e.id AS idstudent',
                DB::raw('CONCAT(ti.tipo, " - ", u.nroidenti) AS identificacion'),
                DB::raw('CONCAT(u.apellido_pa, " ", u.apellido_ma, ", ", u.nombres) AS estudiante')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->join('matriculas_semestres AS ms', 'ms.student_id', '=', 'e.id')
            ->join('semestres AS se', 'se.id', '=', 'ms.semester_id')
            ->where('e.id', '=', $idstudent)
            ->where('se.id', '=', $idsemester)
            ->get();

        // Subconsulta para ver cada nota
        $gradesQuery = DB::table('matriculas_asignaturas AS ma')
            ->selectRaw('pl.nombre AS plan,
            c.codcurso AS codcurso,
            c.nombre AS curso,
            CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) AS creditos_horas,
            pe.numero AS periodo,
            a.tipo AS condicion,
            a.seccion AS seccion,
            a.turno AS turno,
            i.id AS idindicator,
            ac.id AS idactivity,
            n.id AS idgrade,
            ne.id AS idstudentgrade,
            n.nombre AS tipo_nota,
            n.porcentaje AS porcentaje,
            COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 1), 0) AS nota')
            ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
            ->join('cursos AS c', 'c.id', '=', 'a.course_id')
            ->join('periodos AS pe', 'pe.id', '=', 'c.period_id')
            ->join('planes AS pl', 'pl.id', '=', 'pe.plan_id')
            ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
            ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
            ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
            ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
            ->leftJoin('notas_estudiantes AS ne', function ($join) {
                $join->on('ne.grade_id', '=', 'n.id')
                    ->on('ne.subjectenrollment_id', '=', 'ma.id');
            })
            ->where('ma.student_id', '=', $idstudent)
            ->groupBy(
                'plan',
                'curso',
                'codcurso',
                'creditos_horas',
                'periodo',
                'condicion',
                'seccion',
                'turno',
                'idindicator',
                'idactivity',
                'idgrade',
                'idstudentgrade',
                'tipo_nota',
                'porcentaje'
            );

        // Subconsulta para promedios por actividad
        $activityQuery = DB::table(DB::raw('('.$gradesQuery->toSql().') AS subconsulta_2'))
            ->mergeBindings($gradesQuery)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'idindicator', 'idactivity')
            ->selectRaw('plan,
                codcurso,
                curso,
                creditos_horas,
                periodo,
                condicion,
                seccion,
                turno,
                idindicator,
                idactivity,
                COALESCE(ROUND(SUM(nota * porcentaje), 1), 0) AS promedio_actividades');

        // Subconsulta para promedios por indicador
        $indicatorQuery = DB::table(DB::raw('('.$activityQuery->toSql().') AS subconsulta_3'))
            ->mergeBindings($activityQuery)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'idindicator')
            ->selectRaw('plan,
            codcurso,
            curso,
            creditos_horas,
            periodo,
            condicion,
            seccion,
            turno,
            idindicator,
            COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 1), 0) AS promedio_indicadores');

        // Consulta final
        $asignaturas = DB::table(DB::raw('('.$indicatorQuery->toSql().') AS subconsulta_4'))
            ->mergeBindings($indicatorQuery)
            ->groupBy('plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno')
            ->selectRaw('plan,
            codcurso,
            curso,
            creditos_horas,
            periodo,
            condicion,
            seccion,
            turno,
            COALESCE(ROUND(SUM(promedio_indicadores) / COUNT(idindicator), 1), 0) AS promedio_final')
            ->get();

        $pdf = PDF::loadView('reportes.promedios_asignaturas', [
            // 'institutions' => $this->institutions,
            'institutions' => AppServiceProvider::$institutions,
            'infos' => $infos,
            'asignaturas' => $asignaturas,
        ]);

        return $pdf->stream();
    }

    public function list_egresados($idplan)
    {
        $estudiantes = Student::join('usuarios as us', 'us.id', '=', 'estudiantes.user_id')
            ->join('tipos_identificaciones AS it', 'it.id', '=', 'us.identificationtype_id')
            ->select(
                'it.id AS ididentificationtype',
                'it.tipo AS tipoidenti',
                DB::raw('CONCAT(it.tipo, " - ", us.nroidenti) AS identificacion'),
                DB::raw('CONCAT(us.apellido_pa, " ", us.apellido_ma, " ", us.nombres) AS estudiante'),
                'us.correo',
                'us.celular',
                'estudiantes.anho_ingreso'
            )
            ->where('estudiantes.plan_id', $idplan)
            ->where('estudiantes.estado', 5)
            ->get();

        $infos = Plan::join('programas', 'planes.program_id', '=', 'programas.id')
            ->select(
                'planes.nombre as plan',
                'planes.tipo',
                'planes.modalidad',
                'planes.enfoque',
                'programas.codprograma',
                'programas.nombre as programa',
                'programas.nivel_formativo'
            )
            ->where('planes.id', $idplan)
            ->get();

        $pdf = PDF::loadView('reportes.reporte_egresados', [
            'institutions' => AppServiceProvider::$institutions,
            'estudiantes' => $estudiantes,
            'infos' => $infos,
        ]);

        return $pdf->stream();
    }

    public function boleta_pago($idpago)
    {

        $inst = DB::table('info_institucional')
            ->select('nombre', 'logo', 'direccion', 'contacto')
            ->first();

        $institutions = [
            'nombre' => $inst->nombre,
            'direccion' => $inst->direccion,
        ];

        $logoUrl = url('/logo/images/institucion/'.$inst->logo);

        DB::statement("SET lc_time_names = 'es_ES'");

        $infos = DB::table('pagos as pa')
            ->select(
                'pa.id as idpago',
                'pa.fecha',
                DB::raw('DATE_FORMAT(pa.fecha, "%W, %d de %M del %Y") as fecha2'),
                'pa.documento',
                'pa.nro_operacion',
                'pa.monto_inicial',
                'pa.descuento',
                'pa.monto_final',
                'pa.estado',
                'es.id as idest',
                'uses.id as idusuario_est',
                'tiid_est.tipo as tipoidenti_est',
                'uses.nroidenti as nroidenti_est',
                'uses.nombres as nombres_est',
                'uses.apellido_pa as apellido_pa_est',
                'uses.apellido_ma as apellido_ma_est',
                DB::raw('CONCAT(uses.apellido_pa, " ", uses.apellido_ma, ", ", uses.nombres) as estudiante'),
                'uses.telefono as telefono_est',
                'uses.celular as celular_est',
                'uses.correo as correo_est',
                'pr.nombre as programa',
                'ad.id as idadmin',
                'usad.id as idusuario_adm',
                DB::raw('CONCAT(tiid_est.tipo, " - ", uses.nroidenti) AS identificacion'),
                'tiid_adm.tipo as tipoidenti_adm',
                'usad.nroidenti as nroidenti_adm',
                'usad.nombres as nombres_adm',
                'usad.apellido_pa as apellido_pa_adm',
                'usad.apellido_ma as apellido_ma_adm',
                'usad.telefono as telefono_adm',
                'usad.celular as celular_adm',
                'usad.correo as correo_adm',
                'co.id as idconcepto',
                'co.nombre as nombreconcepto',
                'mepa.id as idmetodopago',
                'mepa.nombre as nombremetodopago',
                'ba.id as idbanco',
                'ba.nombre as nombrebanco'
            )
            ->join('estudiantes as es', 'pa.student_id', '=', 'es.id')
            ->join('planes as pl', 'pl.id', '=', 'es.plan_id')
            ->join('programas as pr', 'pr.id', '=', 'pl.program_id')
            ->join('usuarios as uses', 'es.user_id', '=', 'uses.id')
            ->join('tipos_identificaciones as tiid_est', 'uses.identificationtype_id', '=', 'tiid_est.id')
            ->join('administradores as ad', 'pa.administrator_id', '=', 'ad.id')
            ->join('usuarios as usad', 'ad.user_id', '=', 'usad.id')
            ->join('tipos_identificaciones as tiid_adm', 'usad.identificationtype_id', '=', 'tiid_adm.id')
            ->join('conceptos as co', 'pa.concepto_id', '=', 'co.id')
            ->leftJoin('metodos_pagos as mepa', 'pa.metodo_pago_id', '=', 'mepa.id')
            ->leftJoin('bancos as ba', 'pa.banco_id', '=', 'ba.id')
            ->where('pa.id', '=', $idpago)
            ->get();

        $logoPath = null;

        if (! empty($inst->logo)) {
            // Construir la ruta física igual que en verLogo()
            $logoPath = storage_path('app/public/'.$inst->logo);

            // Verificar si existe
            if (! file_exists($logoPath)) {
                Log::warning('Logo no encontrado en PDF: '.$logoPath);
                $logoPath = null;
            }

            $pdf = PDF::loadView('reportes.boleta_pago', [
                'institutions' => AppServiceProvider::$institutions,
                'infos' => $infos,
            ]);

            // Define el tamaño del papel como boleta (ancho: 8 cm, alto: 21 cm)
            $pdf->setPaper([0, 0, 226.77, 566.93], 'portrait');

            return $pdf->stream();
        }
    }

    public function certificado_general($idstudent)
    {

        DB::statement("SET lc_time_names = 'es_ES'");

        $inst = DB::table('info_institucional')
            ->select('nombre', 'logo', 'direccion', 'contacto')
            ->first();

        $institutions = [
            'nombre' => $inst->nombre,
            'direccion' => $inst->direccion,
        ];

        $logoUrl = url('/logo/images/institucion/'.$inst->logo);

        // Subconsulta gradesQuery
        $gradesQuery = DB::table('estudiantes AS es')
            ->join('matriculas_semestres AS ms', 'ms.student_id', '=', 'es.id')
            ->join('matriculas_asignaturas AS ma', 'ma.student_id', '=', 'ms.student_id')
            ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
            ->join('cursos AS c', 'c.id', '=', 'a.course_id')
            ->join('periodos AS pe', 'pe.id', '=', 'c.period_id')
            ->join('planes AS pl', 'pl.id', '=', 'pe.plan_id')
            ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
            ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
            ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
            ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
            ->leftJoin('notas_estudiantes AS ne', function ($join) {
                $join->on('ne.grade_id', '=', 'n.id')
                    ->on('ne.subjectenrollment_id', '=', 'ma.id');
            })
            ->selectRaw('
                pl.nombre AS plan,
                c.codcurso AS codcurso,
                c.nombre AS curso,
                CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) AS creditos_horas,
                pe.numero AS periodo,
                a.tipo AS condicion,
                a.seccion AS seccion,
                a.turno AS turno,
                CONCAT(se.anho, " - ", se.numero) AS semestre,
                se.fecinicio AS fecha_semestre,
                i.id AS idindicator,
                ac.id AS idactivity,
                n.id AS idgrade,
                ne.id AS idstudentgrade,
                n.nombre AS tipo_nota,
                n.porcentaje AS porcentaje,
                COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 1), 0) AS nota
            ')
            ->where('ma.student_id', '=', $idstudent)
            ->groupBy([
                'plan',
                'curso',
                'codcurso',
                'creditos_horas',
                'periodo',
                'condicion',
                'seccion',
                'turno',
                'semestre',
                'fecha_semestre',
                'idindicator',
                'idactivity',
                'idgrade',
                'idstudentgrade',
                'tipo_nota',
                'porcentaje',
            ])
            ->orderBy('curso');

        // Subconsulta activityQuery
        $activityQuery = DB::table(DB::raw("({$gradesQuery->toSql()}) AS gradesQuery"))
            ->mergeBindings($gradesQuery)
            ->selectRaw('
                plan,
                codcurso,
                curso,
                creditos_horas,
                periodo,
                condicion,
                seccion,
                turno,
                semestre,
                fecha_semestre,
                idindicator,
                idactivity,
                COALESCE(ROUND(SUM(nota * porcentaje), 1), 0) AS promedio_actividades
            ')
            ->groupBy([
                'plan',
                'curso',
                'codcurso',
                'creditos_horas',
                'periodo',
                'condicion',
                'seccion',
                'turno',
                'semestre',
                'fecha_semestre',
                'idindicator',
                'idactivity',
            ]);

        // Subconsulta indicatorQuery
        $indicatorQuery = DB::table(DB::raw("({$activityQuery->toSql()}) AS activityQuery"))
            ->mergeBindings($activityQuery)
            ->selectRaw('
                plan,
                codcurso,
                curso,
                creditos_horas,
                periodo,
                condicion,
                seccion,
                turno,
                semestre,
                fecha_semestre,
                idindicator,
                COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 1), 0) AS promedio_indicadores
            ')
            ->groupBy([
                'plan',
                'curso',
                'codcurso',
                'creditos_horas',
                'periodo',
                'condicion',
                'seccion',
                'turno',
                'semestre',
                'fecha_semestre',
                'idindicator',
            ]);

        // Subconsulta finalGrades
        $finalGrades = DB::table(DB::raw("({$indicatorQuery->toSql()}) AS indicatorQuery"))
            ->mergeBindings($indicatorQuery)
            ->selectRaw('
                plan,
                codcurso,
                curso,
                creditos_horas,
                periodo,
                condicion,
                seccion,
                turno,
                semestre,
                fecha_semestre,
                COALESCE(ROUND(SUM(promedio_indicadores) / COUNT(idindicator), 1), 0) AS promedio_final
            ')
            ->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre', 'fecha_semestre']);

        // Consulta final
        $cursos = DB::table('planes AS pl')
            ->join('estudiantes AS es', 'es.plan_id', '=', 'pl.id')
            ->join('periodos AS pe', 'pe.plan_id', '=', 'pl.id')
            ->join('cursos AS c', 'c.period_id', '=', 'pe.id')
            ->join('modulos_formativos AS mf', 'mf.id', '=', 'c.training_module_id')
            ->leftJoinSub($finalGrades, 'fg', function ($join) {
                $join->on('fg.codcurso', '=', 'c.codcurso')
                    ->on('fg.plan', '=', 'pl.nombre');
            })
            ->selectRaw('
                pl.nombre AS plan,
                mf.numero AS modulo,
                pe.numero AS periodo,
                c.codcurso AS codcurso,
                c.nombre AS nombre_curso,
                c.creditos AS creditos,
                c.horas AS horas,
                fg.seccion AS seccion,
                fg.turno AS turno,
                DATE_FORMAT(fg.fecha_semestre, "%d/%m/%Y") as fecha_semestre,
                COALESCE (fg.semestre, "--") AS semestre,
                COALESCE(fg.promedio_final, "--") AS nota_numero,
                CASE
                    WHEN fg.promedio_final = 0 THEN "cero"
                    WHEN fg.promedio_final = 1 THEN "uno"
                    WHEN fg.promedio_final = 2 THEN "dos"
                    WHEN fg.promedio_final = 3 THEN "tres"
                    WHEN fg.promedio_final = 4 THEN "cuatro"
                    WHEN fg.promedio_final = 5 THEN "cinco"
                    WHEN fg.promedio_final = 6 THEN "seis"
                    WHEN fg.promedio_final = 7 THEN "siete"
                    WHEN fg.promedio_final = 8 THEN "ocho"
                    WHEN fg.promedio_final = 9 THEN "nueve"
                    WHEN fg.promedio_final = 10 THEN "diez"
                    WHEN fg.promedio_final = 11 THEN "once"
                    WHEN fg.promedio_final = 12 THEN "doce"
                    WHEN fg.promedio_final = 13 THEN "trece"
                    WHEN fg.promedio_final = 14 THEN "catorce"
                    WHEN fg.promedio_final = 15 THEN "quince"
                    WHEN fg.promedio_final = 16 THEN "dieciséis"
                    WHEN fg.promedio_final = 17 THEN "diecisiete"
                    WHEN fg.promedio_final = 18 THEN "dieciocho"
                    WHEN fg.promedio_final = 19 THEN "diecinueve"
                    WHEN fg.promedio_final = 20 THEN "veinte"
                    ELSE "--"
                END AS nota_letras,
                CASE
                    WHEN fg.promedio_final >= 13 THEN "Aprobado"
                    WHEN fg.promedio_final < 13 AND fg.promedio_final > 0 THEN "Desaprobado"
                    WHEN fg.promedio_final = 0 AND fg.plan IS NOT NULL THEN "Por evaluar"
                    ELSE "--"
                END AS estado
            ')
            ->orderBy('semestre', 'desc')
            ->orderBy('periodo')
            ->orderBy('curso')
            ->where('es.id', '=', $idstudent)
            ->get();

        $estudiantes = DB::table('estudiantes AS e')
            ->select(
                'e.id AS idstudent',
                'pl.nombre AS plan',
                'pr.nombre AS programa',
                DB::raw('DATE_FORMAT(NOW(), "%e de %M del %Y") as fecha_actual'),
                DB::raw('CONCAT(ti.tipo, " - ", u.nroidenti) AS identificacion'),
                DB::raw('CONCAT(u.nombres, " ", u.apellido_pa, " ", u.apellido_ma) AS estudiante')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->where('e.id', '=', $idstudent)
            ->groupBy('idstudent', 'estudiante', 'identificacion', 'plan', 'programa')
            ->first();

        $logoPath = null;

        if (! empty($inst->logo)) {
            // Construir la ruta física igual que en verLogo()
            $logoPath = storage_path('app/public/'.$inst->logo);

            // Verificar si existe
            if (! file_exists($logoPath)) {
                Log::warning('Logo no encontrado en PDF: '.$logoPath);
                $logoPath = null;
            }
            $mineduLogoPath = public_path('images/institucion/minedu.jpg');
            if (! file_exists($mineduLogoPath)) {
                Log::warning('Logo MINEDU no encontrado en: '.$mineduLogoPath);
                $mineduLogoPath = null;
            }

            $pdf = PDF::loadView('reportes.certificado_estudios', [
                'institutions' => $institutions,
                'logoSrc' => $logoPath,
                'estudiantes' => $estudiantes,
                'mineduLogoSrc' => $mineduLogoPath,
                'cursos' => $cursos,
            ]);

            $pdf->setPaper('A4', 'portrait');

            return $pdf->stream();
        }
    }

    public function certificado_modular($idstudent, $idmodulo)
    {
        $existingModularCertificate = DB::table('certificados_modulares')
            ->where('student_id', $idstudent)
            ->where('training_module_id', $idmodulo)
            ->first();

        if ($existingModularCertificate && Storage::disk('public')->exists($existingModularCertificate->certificado)) {
            $pathCompleto = storage_path('app/public/'.$existingModularCertificate->certificado);
            $filename = 'modulo-'.$idmodulo.'.pdf';

            return response()->file($pathCompleto, [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'inline; filename="'.$filename.'"',
            ]);
        }

        $inst = DB::table('info_institucional')->select('nombre', 'logo', 'direccion')->first();
        $institutions = ['nombre' => $inst->nombre ?? '', 'direccion' => $inst->direccion ?? ''];

        $estudiante_data = DB::table('estudiantes AS e')
            ->select(
                'e.id AS idstudent',
                'pl.nombre AS plan',
                'pr.nombre AS programa',
                DB::raw('CONCAT(ti.tipo, " - ", u.nroidenti) AS identificacion'),
                DB::raw('CONCAT(u.nombres, " ", u.apellido_pa, " ", u.apellido_ma) AS estudiante'),
                'mf.id as idmodulo',
                'mf.numero as numero',
                'mf.codigo as codigo',
                'mf.descripcion as descripcion',
                DB::raw('COALESCE(SUM(CASE WHEN tr.id IS NOT NULL THEN cc.creditos ELSE c.creditos END), "--") AS total_creditos'),
                DB::raw('COALESCE(SUM(CASE WHEN tr.id IS NOT NULL THEN cc.horas_por_semana ELSE c.horas END), "--") AS total_horas'),
                DB::raw('SUM(CASE WHEN tr.id IS NOT NULL THEN 1 ELSE 0 END) as total_cursos_convalidados'),
                DB::raw('COALESCE(SUM(CASE WHEN tr.id IS NOT NULL THEN cc.creditos ELSE 0 END), 0) as total_creditos_convalidados'),
                DB::raw('COALESCE(SUM(CASE WHEN tr.id IS NOT NULL THEN cc.horas_por_semana ELSE 0 END), 0) as total_horas_convalidadas'),
                DB::raw('COALESCE(MIN(DATE_FORMAT(s.fecinicio, "%e de %M del %Y")), "--") as fecha_ini'),
                DB::raw('COALESCE(MIN(DATE_FORMAT(s.fecfin, "%e de %M del %Y")), "--") as fecha_fin'),
                DB::raw('COALESCE(DATE_FORMAT(NOW(), "%e de %M del %Y"), "--") as fecha_actual')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('modulos_formativos AS mf', 'mf.id', '=', DB::raw($idmodulo))
            ->join('planes AS pl', 'pl.id', '=', 'mf.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->join('cursos AS c', 'c.training_module_id', '=', 'mf.id')
            ->leftJoin('convalidacion_cursos as cc', 'c.id', '=', 'cc.course_id')
            ->leftJoin('convalidaciones as ce', 'cc.convalidation_id', '=', 'ce.id')
            ->leftJoin('traslados as tr', function ($join) {
                $join->on('ce.transfers_id', '=', 'tr.id')
                    ->on('tr.student_id', '=', 'e.id');
            })

            ->leftJoin('matriculas_asignaturas as ma', function ($join) {
                $join->on('ma.student_id', '=', 'e.id')
                    ->whereRaw('ma.subject_id IN (SELECT id FROM asignaturas WHERE course_id = c.id)');
            })
            ->leftJoin('asignaturas as a', 'a.id', '=', 'ma.subject_id')
            ->leftJoin('semestres as s', 's.id', '=', 'a.semester_id')
            ->leftJoin('matriculas_semestres as ms', function ($join) {
                $join->on('ms.student_id', '=', 'e.id')
                    ->on('ms.semester_id', '=', 's.id');
            })
            ->where('e.id', '=', $idstudent)
            ->groupBy(
                'idstudent', 'estudiante', 'identificacion', 'plan', 'programa',
                'mf.id', 'mf.numero', 'mf.codigo', 'mf.descripcion'
            )
            ->orderByRaw('MIN(s.fecinicio) ASC')
            ->first();

        // Validación si no hay datos
        if (! $estudiante_data) {
            return response('Error: No se encontraron datos académicos para el estudiante o módulo especificado. Verifique que el módulo tenga cursos y el estudiante notas o convalidaciones.', 404);
        }

        // Formateo final de fecha actual si vino nula de la BD
        if ($estudiante_data->fecha_actual == '--') {
            $estudiante_data->fecha_actual = Carbon::now()->locale('es')->isoFormat('D [de] MMMM [del] YYYY');
        }

        // 4. Generación del Código de Verificación
        $existingValidation = DB::table('certificados_validados')
            ->where('student_id', $idstudent)
            ->where('training_module_id', $idmodulo)
            ->first();

        $verificationCode = null;
        $verificationId = null;

        if ($existingValidation) {
            $verificationCode = $existingValidation->codigo_verificacion;
            $verificationId = $existingValidation->id;
        } else {
            try {
                $verificationCode = Str::uuid()->toString();
                $verificationId = DB::table('certificados_validados')->insertGetId([
                    'student_id' => $idstudent,
                    'training_module_id' => $idmodulo,
                    'codigo_verificacion' => $verificationCode,
                    'issued_at' => now(),
                    'module_description' => $estudiante_data->descripcion,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
            } catch (\Exception $e) {
                Log::error('Error al guardar validación: '.$e->getMessage());

                return response('Error interno al registrar la verificación.', 500);
            }
        }

        // 5. Generación del QR
        $verificationUrl = route('verification.show', ['code' => $verificationCode]);
        $qrCodeSrc = null;

        try {
            $result = Builder::create()
                ->writer(new PngWriter())
                ->data($verificationUrl)
                ->encoding(new Encoding('UTF-8'))
                ->errorCorrectionLevel(ErrorCorrectionLevel::High)
                ->size(240)
                ->margin(10)
                ->build();

            if ($result instanceof GdResult) {
                $qrCodeSrc = 'data:image/png;base64,'.base64_encode($result->getString());
            } else {
                $qrCodeSrc = $result->getDataUri();
            }
        } catch (\Throwable $e) {
            Log::error('Error QR: '.$e->getMessage());
        }

        // 6. Carga de Logos
        $logoPath = null;
        if (! empty($inst->logo)) {
            $candidate = storage_path('app/public/'.$inst->logo);
            if (file_exists($candidate)) {
                $logoPath = $candidate;
            }
        }

        $mineduLogoPath = public_path('images/institucion/minedu.jpg');
        if (! file_exists($mineduLogoPath)) {
            $mineduLogoPath = null;
        }

        // 7. Renderizado del PDF
        $pdf = PDF::loadView('reportes.certificado_modular', [
            'institutions' => $institutions,
            'logoSrc' => $logoPath,
            'mineduLogoSrc' => $mineduLogoPath,
            'estudiantes' => $estudiante_data,
            'qrCodeSrc' => $qrCodeSrc,
        ]);

        $pdf->setPaper('A4', 'landscape');

        // 8. Guardado y Registro en BD
        try {
            $filename = 'modulo-'.$idmodulo.'.pdf';
            $path = 'students/'.$idstudent.'/certificados_modulares/'.$filename;

            // Guardar archivo físico
            Storage::disk('public')->put($path, $pdf->output());

            // Verificar si ya existe el registro del certificado
            $existsCert = DB::table('certificados_modulares')
                ->where('student_id', $idstudent)
                ->where('training_module_id', $idmodulo) // <--- Nombre correcto según imagen 1
                ->first();

            // Datos a guardar (Coincidiendo con tu Imagen 1)
            $datosCertificado = [
                'certificado' => $path,
                'validated_certificate_id' => $verificationId, // <--- Clave foránea correcta
                'updated_at' => Carbon::now(),
            ];

            if ($existsCert) {
                // ACTUALIZAR (UPDATE)
                DB::table('certificados_modulares')
                    ->where('id', $existsCert->id)
                    ->update($datosCertificado);
            } else {
                // INSERTAR (INSERT)
                // Agregamos los campos que faltan para el insert
                $datosCertificado['student_id'] = $idstudent;
                $datosCertificado['training_module_id'] = $idmodulo; // <--- Nombre correcto
                $datosCertificado['created_at'] = Carbon::now();

                DB::table('certificados_modulares')->insert($datosCertificado);
            }

        } catch (\Exception $e) {
            Log::error('Error al guardar en certificados_modulares: '.$e->getMessage());

            return $pdf->stream('certificado-no-guardado.pdf');
        }

        return $pdf->stream($filename);

        return $pdf->stream($filename);
    }

    // Función para mostrar el certificado público basado en el código de verificación
    public function MostrarCertificadoModular(string $code)
    {
        $validationRecord = DB::table('certificados_validados')
            ->where('codigo_verificacion', $code)
            ->first();
        if (! $validationRecord) {
            return response('Código de verificación inválido o no encontrado.', 404);
        }
        $cert = DB::table('certificados_modulares')
            ->where('validated_certificate_id', $validationRecord->id)
            ->orderByDesc('updated_at')
            ->first();
        if (! $cert) {
            Log::error("Registro de validación (ID: {$validationRecord->id}) encontrado, pero no se encontró un certificado modular asociado.");

            return response('No se encontró un certificado modular asociado a esta validación.', 404);
        }
        if (! Storage::disk('public')->exists($cert->certificado)) {
            Log::error("Archivo de certificado (ID: {$cert->id}) no encontrado en storage. Ruta: {$cert->certificado}");

            return response('El archivo del certificado no está disponible o ha sido movido.', 404);
        }
        $absolutePath = storage_path('app/public/'.$cert->certificado);

        return response()->file($absolutePath, ['Content-Type' => 'application/pdf']);
    }

    public function constancia_egresado($idstudent)
    {

        $inst = DB::table('info_institucional')
            ->select('nombre', 'logo', 'direccion', 'contacto')
            ->first();

        $institutions = [
            'nombre' => $inst->nombre,
            'direccion' => $inst->direccion,
        ];

        $estudiantes = DB::table('estudiantes AS e')
            ->select(
                'e.id AS idstudent',
                'pl.nombre AS plan',
                'pr.nombre AS programa',
                'pr.nivel_formativo AS nivel',
                'u.nroidenti AS identificacion',
                DB::raw('CONCAT(u.nombres, " ", u.apellido_pa, " ", u.apellido_ma) AS estudiante'),
                DB::raw('COALESCE(DATE_FORMAT(NOW(), "%e de %M del %Y"), "--") AS fecha_actual')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->where('e.id', '=', $idstudent)
            ->groupBy('e.id', 'u.nroidenti', 'estudiante', 'nivel', 'pl.nombre', 'pr.nombre')
            ->first();

        if (! empty($inst->logo)) {
            $logoPath = storage_path('app/public/'.$inst->logo);

            if (! file_exists($logoPath)) {
                Log::warning('Logo no encontrado en PDF: '.$logoPath);
                $logoPath = null;
            }

            $mineduLogoPath = public_path('images/institucion/minedu.jpg');
            if (! file_exists($mineduLogoPath)) {
                Log::warning('Logo MINEDU no encontrado en: '.$mineduLogoPath);
                $mineduLogoPath = null;
            }
            $pdf = PDF::loadView('reportes.constancia_egresado', [
                'institutions' => $institutions,
                'logoSrc' => $logoPath,
                'mineduLogoSrc' => $mineduLogoPath,
                'estudiantes' => $estudiantes,
            ]);

            $pdf->setPaper('A4', 'portrait');

            return $pdf->stream();
        }
    }

    public function diploma_estudios($idstudent)
    {

        $inst = DB::table('info_institucional')
            ->select('nombre', 'logo', 'direccion', 'contacto')
            ->first();

        $institutions = [
            'nombre' => $inst->nombre,
            'direccion' => $inst->direccion,
        ];

        DB::statement("SET lc_time_names = 'es_ES'");

        $estudiantes = DB::table('estudiantes AS e')
            ->select(
                'e.id AS idstudent',
                'e.anho_ingreso',
                'pl.nombre AS plan',
                'pr.nombre AS programa',
                'pr.nivel_formativo AS nivel',
                'u.nroidenti AS identificacion',
                DB::raw('CONCAT(u.nombres, " ", u.apellido_pa, " ", u.apellido_ma) AS estudiante'),
                DB::raw('COALESCE(DATE_FORMAT(NOW(), "%e de %M del %Y"), "--") AS fecha_actual')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->where('e.id', '=', $idstudent)
            ->groupBy('e.id', 'u.nroidenti', 'estudiante', 'nivel', 'pl.nombre', 'pr.nombre')
            ->first();

        if (! empty($inst->logo)) {
            // Construir la ruta física igual que en verLogo()
            $logoPath = storage_path('app/public/'.$inst->logo);

            // Verificar si existe
            if (! file_exists($logoPath)) {
                Log::warning('Logo no encontrado en PDF: '.$logoPath);
                $logoPath = null;
            }

            $pdf = PDF::loadView('reportes.constancia_egresado', [
                'institutions' => $institutions,
                'logoSrc' => $logoPath,
                'estudiantes' => $estudiantes,
            ]);
            $pdf = PDF::loadView('reportes.diploma_estudios', [
                'institutions' => $institutions,
                'logoSrc' => $logoPath,
                'estudiantes' => $estudiantes,
            ]);

            $pdf->setPaper('A4', 'landscape');

            return $pdf->stream();
        }
    }

    public function prueba()
    {
        $idstudent = 1;

        $estudiantes = DB::table('estudiantes AS e')
            ->select(
                'e.id AS idstudent',
                'e.anho_ingreso',
                'pl.nombre AS plan',
                'pr.nombre AS programa',
                'pr.nivel_formativo AS nivel',
                'u.nroidenti AS identificacion',
                DB::raw('CONCAT(u.nombres, " ", u.apellido_pa, " ", u.apellido_ma) AS estudiante'),
                DB::raw('COALESCE(DATE_FORMAT(NOW(), "%e de %M del %Y"), "--") AS fecha_actual')
            )
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->where('e.id', '=', $idstudent)
            ->groupBy('e.id', 'u.nroidenti', 'estudiante', 'nivel', 'pl.nombre', 'pr.nombre')
            ->first();

        $pdf = PDF::loadView('reportes.prueba', [
            'institutions' => AppServiceProvider::$institutions,
            'estudiantes' => $estudiantes,
        ]);

        $pdf->setPaper('A4', 'landscape');

        return $pdf->stream();
    }

    public function record_notas($idstudent)
    {
        $inst = DB::table('info_institucional')
            ->select('nombre', 'logo', 'direccion')
            ->first();

        $institutions = [
            'nombre' => $inst->nombre ?? 'Nombre no disponible',
            'direccion' => $inst->direccion ?? 'Dirección no disponible',
        ];

        $logoPath = null;
        if (! empty($inst->logo)) {
            $potentialPath = storage_path('app/public/'.$inst->logo);
            if (file_exists($potentialPath)) {
                $logoPath = $potentialPath;
            } else {
                Log::warning('Logo de institución no encontrado en PDF (récord de notas): '.$potentialPath);
            }
        }

        $mineduLogoPath = public_path('images/institucion/minedu.jpg');
        if (! file_exists($mineduLogoPath)) {
            Log::warning('Logo MINEDU no encontrado en: '.$mineduLogoPath);
            $mineduLogoPath = null;
        }

        $gradesQuery = DB::table('estudiantes AS es')
            ->join('matriculas_semestres AS ms', 'ms.student_id', '=', 'es.id')
            ->join('matriculas_asignaturas AS ma', 'ma.student_id', '=', 'ms.student_id')
            ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
            ->join('cursos AS c', 'c.id', '=', 'a.course_id')
            ->join('periodos AS pe', 'pe.id', '=', 'c.period_id')
            ->join('planes AS pl', 'pl.id', '=', 'pe.plan_id')
            ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
            ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
            ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
            ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
            ->leftJoin('notas_estudiantes AS ne', function ($join) {
                $join->on('ne.grade_id', '=', 'n.id')
                    ->on('ne.subjectenrollment_id', '=', 'ma.id');
            })
            ->selectRaw('es.id AS idstudent, pl.nombre AS plan, c.codcurso AS codcurso, c.nombre AS curso, CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) AS creditos_horas, pe.numero AS periodo, a.tipo AS condicion, a.seccion AS seccion, a.turno AS turno, CONCAT(se.anho, " - ", se.numero) AS semestre, i.id AS idindicator, ac.id AS idactivity, n.id AS idgrade, ne.id AS idstudentgrade, n.nombre AS tipo_nota, n.porcentaje AS porcentaje, COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 1), 0) AS nota')
            ->where('ma.student_id', '=', $idstudent)
            ->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre', 'idindicator', 'idactivity', 'idgrade', 'idstudentgrade', 'tipo_nota', 'porcentaje', 'idstudent'])
            ->orderBy('curso');

        $activityQuery = DB::table(DB::raw("({$gradesQuery->toSql()}) AS gradesQuery"))->mergeBindings($gradesQuery)->selectRaw('plan, codcurso, curso, creditos_horas, periodo, condicion, seccion, turno, semestre, idindicator, idactivity, COALESCE(ROUND(SUM(nota * porcentaje), 1), 0) AS promedio_actividades')->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre', 'idindicator', 'idactivity']);
        $indicatorQuery = DB::table(DB::raw("({$activityQuery->toSql()}) AS activityQuery"))->mergeBindings($activityQuery)->selectRaw('plan, codcurso, curso, creditos_horas, periodo, condicion, seccion, turno, semestre, idindicator, COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 1), 0) AS promedio_indicadores')->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre', 'idindicator']);
        $finalGrades = DB::table(DB::raw("({$indicatorQuery->toSql()}) AS indicatorQuery"))->mergeBindings($indicatorQuery)->selectRaw('plan, codcurso, curso, creditos_horas, periodo, condicion, seccion, turno, semestre, COALESCE(ROUND(SUM(promedio_indicadores) / COUNT(idindicator), 1), 0) AS promedio_final')->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre']);

        $cursos = DB::table('planes AS pl')
            ->join('estudiantes AS es', 'es.plan_id', '=', 'pl.id')
            ->join('periodos AS pe', 'pe.plan_id', '=', 'pl.id')
            ->join('cursos AS c', 'c.period_id', '=', 'pe.id')
            ->leftJoinSub($finalGrades, 'fg', function ($join) {
                $join->on('fg.codcurso', '=', 'c.codcurso')->on('fg.plan', '=', 'pl.nombre');
            })
            ->selectRaw('fg.semestre AS semestre, pe.numero AS ciclo, c.codcurso AS codcurso, c.nombre AS nombre_curso, c.horas AS horas, c.creditos AS creditos, fg.promedio_final AS promedio_final')
            ->orderBy('ciclo')->orderBy('nombre_curso')
            ->where('es.id', '=', $idstudent)
            ->get();

        $estudiantes = DB::table('estudiantes AS e')
            ->select('e.id AS idstudent', 'pl.nombre AS plan', 'pr.nombre AS programa', DB::raw('DATE_FORMAT(NOW(), "%e de %M del %Y") as fecha_actual'), DB::raw('CONCAT(ti.tipo, " - ", u.nroidenti) AS identificacion'), DB::raw('CONCAT(u.nombres, " ", u.apellido_pa, " ", u.apellido_ma) AS estudiante'))
            ->join('usuarios AS u', 'u.id', '=', 'e.user_id')
            ->join('tipos_identificaciones AS ti', 'ti.id', '=', 'u.identificationtype_id')
            ->join('planes AS pl', 'pl.id', '=', 'e.plan_id')
            ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
            ->where('e.id', '=', $idstudent)
            ->groupBy('idstudent', 'estudiante', 'identificacion', 'plan', 'programa')
            ->first();

        if (! $estudiantes) {
            return response("No se encontraron datos para el estudiante con ID: $idstudent", 404);
        }

        $pdf = PDF::loadView('reportes.record_notas', [
            'institutions' => $institutions,
            'cursos' => $cursos,
            'estudiantes' => $estudiantes,
            'logoSrc' => $logoPath,
            'mineduLogoSrc' => $mineduLogoPath,
        ]);

        $pdf->setPaper('A4', 'portrait');

        return $pdf->stream('record-notas.pdf');
    }

    public function validarCertificadoParaEmision($idstudent)
    {
        $academicErrors = $this->verificarElegibilidadCertificado($idstudent);
        $financialErrors = $this->verificarDeudasPendientes($idstudent);

        $allErrors = [];
        if (! empty($academicErrors)) {
            $allErrors['academic'] = $academicErrors;
        }
        if (! empty($financialErrors)) {
            $allErrors['financial'] = $financialErrors;
        }

        if (! empty($allErrors)) {
            return response()->json([
                'status' => false,
                'errors' => $allErrors,
            ]);
        }

        return response()->json(['status' => true, 'message' => 'Estudiante elegible.']);
    }

    private function verificarDeudasPendientes($idstudent)
    {
        $tienePagosPendientes = DB::table('pagos')
            ->where('student_id', $idstudent)
            ->whereIn('estado', [0, 2]) // 0: Pendiente, 2: Observado
            ->exists();

        if ($tienePagosPendientes) {
            return ['El estudiante presenta deudas o pagos pendientes.'];
        }

        return [];
    }

    private function verificarElegibilidadCertificado($idstudent)
    {
        $sql = "
            WITH ConteoNotas AS (
                SELECT
                    ma.id AS matricula_id,
                    COUNT(DISTINCT n.id) AS total_notas_definidas,
                    COUNT(DISTINCT ne.id) AS total_notas_ingresadas
                FROM matriculas_asignaturas ma
                JOIN asignaturas a ON a.id = ma.subject_id
                LEFT JOIN indicadores i ON i.subject_id = a.id
                LEFT JOIN actividades ac ON ac.indicator_id = i.id
                LEFT JOIN notas n ON n.activity_id = ac.id
                LEFT JOIN notas_estudiantes ne ON ne.grade_id = n.id AND ne.subjectenrollment_id = ma.id
                WHERE ma.student_id = ?
                GROUP BY ma.id
            )
            -- INICIO DE LA CONSULTA PRINCIPAL
            SELECT
                X.curso_nombre,
                MAX(X.nota_minima_requerida) AS nota_minima_requerida,
                CASE
                    WHEN SUM(X.es_cursado) > 0 AND MAX(X.total_notas_definidas) > MAX(X.total_notas_ingresadas)
                    THEN 'SI'
                    ELSE 'NO'
                END AS notas_incompletas,
                CASE
                    WHEN SUM(X.es_cursado) > 0 THEN ROUND(SUM(X.nota_obtenida * (X.porcentaje / 100)), 2)
                    ELSE NULL
                END AS promedio_final
            FROM (
              -- PARTE 1: Cursos cursados regularmente
              SELECT
                c.id AS curso_id,
                c.nombre AS curso_nombre,
                a.nota_minima AS nota_minima_requerida,
                n.porcentaje AS porcentaje,
                COALESCE(ne.nota, 0) AS nota_obtenida,
                -- Unimos los conteos desde la subconsulta
                cn.total_notas_definidas,
                cn.total_notas_ingresadas,
                1 AS es_cursado
              FROM matriculas_asignaturas ma
              JOIN asignaturas a ON a.id = ma.subject_id
              JOIN cursos c ON c.id = a.course_id
              LEFT JOIN notas n ON n.activity_id IN (SELECT ac.id FROM actividades ac JOIN indicadores i ON ac.indicator_id = i.id WHERE i.subject_id = a.id)
              LEFT JOIN notas_estudiantes ne ON ne.grade_id = n.id AND ne.subjectenrollment_id = ma.id
              -- Unimos la subconsulta de conteos
              JOIN ConteoNotas cn ON cn.matricula_id = ma.id
              WHERE ma.student_id = ?

              UNION ALL

              -- PARTE 2: Cursos por convalidación
              SELECT
                c.id, c.nombre, 13, 100, cc.final_average,
                1, 1, 0
              FROM convalidacion_cursos cc
              JOIN convalidaciones_externas ce ON ce.id = cc.convalidation_id
              JOIN cursos c ON c.id = cc.course_id
              WHERE ce.student_id = ? AND cc.deleted_at IS NULL
            ) AS X
            WHERE X.es_cursado = 1
            GROUP BY X.curso_id, X.curso_nombre;
        ";

        // Ejecutamos la consulta. Nota que el ID se pasa 3 veces ahora.
        $resultados = DB::select($sql, [$idstudent, $idstudent, $idstudent]);

        // Bucle para encontrar el PRIMER error
        foreach ($resultados as $curso) {
            if ($curso->notas_incompletas == 'SI' || ($curso->promedio_final !== null && $curso->promedio_final < $curso->nota_minima_requerida)) {
                // Si encuentra CUALQUIER error, devuelve inmediatamente el mensaje genérico.
                return ['El estudiante no cumple con los requisitos académicos para la emisión del documento. Por favor, revise el récord de notas para más detalles.'];
            }
        }

        // Si el bucle termina sin encontrar errores, devuelve un array vacío.
        return [];
    }

    public function validarCertificadoModular($idstudent, $idmodule)
    {

        $sql = '
        SELECT
            c.nombre AS curso_con_nota_faltante
        FROM
            estudiantes e
        JOIN
            planes p ON e.plan_id = p.id
        JOIN
            modulos_formativos mf ON mf.plan_id = p.id
        JOIN
            cursos c ON c.training_module_id = mf.id
        LEFT JOIN
            asignaturas a ON a.course_id = c.id
        LEFT JOIN
            matriculas_asignaturas ma ON ma.subject_id = a.id AND ma.student_id = e.id
        LEFT JOIN
            indicadores i ON i.subject_id = a.id
        LEFT JOIN
            actividades ac ON ac.indicator_id = i.id
        LEFT JOIN
            notas n ON n.activity_id = ac.id
        LEFT JOIN
            notas_estudiantes ne ON ne.grade_id = n.id AND ne.subjectenrollment_id = ma.id
        WHERE
            e.id = ?
            AND mf.id = ?
            AND ne.id IS NULL -- ¡Esta es la condición clave! Busca notas no registradas
            AND n.id IS NOT NULL -- Nos aseguramos de que la evaluación esté definida
        GROUP BY
            c.nombre
        LIMIT 1; -- Solo necesitamos saber si existe al menos una nota faltante
    ';

        $notas_faltantes = DB::select($sql, [$idstudent, $idmodule]);

        if (! empty($notas_faltantes)) {
            $curso_con_error = $notas_faltantes[0]->curso_con_nota_faltante;

            return response()->json([
                'status' => false,
                'errors' => [
                    'academic' => ["Se han encontrado notas faltantes en el curso: '$curso_con_error'. Por favor, complete todas las calificaciones."],
                ],
            ]);
        }

        return response()->json(['status' => true, 'message' => 'Todas las notas del módulo están completas.']);
    }
}
