<?php

namespace App\Http\Controllers\docente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Studentgrade;
use Illuminate\Support\Facades\Validator;
use \Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;
use Yajra\DataTables\Facades\DataTables;
use App\Models\Teacher;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ZoomController extends Controller
{
    /**
     * Obtiene el token de acceso utilizando el flujo Server-to-Server OAuth.
     */
    private function getAccessToken()
    {
        $url = 'https://zoom.us/oauth/token';

        // Obtener el usuario autenticado
        $user = Auth::user();

        if (!$user) {
            return null;  // Devolver null si el usuario no está autenticado
        }

        // Acceder a la relación 'teachers'
        $teacherdata = $user->teachers;

        if (!$teacherdata || $teacherdata->isEmpty()) {
            return null;  // Devolver null si no hay profesores asociados
        }

        $idteacher = $teacherdata[0]->id;

        // Buscar la información del profesor en la base de datos
        $data = Teacher::find($idteacher);

        if (!$data) {
            return null;  // Devolver null si no se encuentra al profesor
        }

        // Validar las credenciales del profesor
        $accountId = $data->account_id;
        $clientId = $data->client_id;
        $clientSecret = $data->client_secret;

        if (is_null($accountId) || is_null($clientId) || is_null($clientSecret)) {
            return null;  // Devolver null si faltan credenciales
        }

        // Crear la cabecera de autorización Basic Auth
        $authorization = base64_encode("{$clientId}:{$clientSecret}");

        // Enviar la solicitud a Zoom
        $response = Http::withHeaders([
            'Authorization' => "Basic $authorization",
        ])->asForm()->post($url, [
            'grant_type' => 'account_credentials',
            'account_id' => $accountId,
        ]);

        // Verificar si la respuesta es exitosa y devolver el token
        if ($response->successful()) {
            return $response->json()['access_token'];
        }

        return null;  // Si no se obtiene el token, devolver null
    }

    /*----------------------------------------------------------------------------------------
     ********************** MÉTODOS DE GESTIÓN DE REUNIÓN PARA DOCENTES **********************
     ******************************* PESTAÑA DE ZOOM DE ALUMNOS ******************************
     *---------------------------------------------------------------------------------------*/
    /**
     * Lista las reuniones del usuario.
     */
    public function listMeetings()
    {
        // Obtiene el token de acceso
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return redirect()->route('docente.zoom.index')->with('error', 'No se pudo obtener el token de acceso.');
        }

        // Realiza la solicitud para obtener las reuniones
        $response = Http::withToken($accessToken)
            ->get('https://api.zoom.us/v2/users/me/meetings');

        // Verifica si la respuesta fue exitosa
        if ($response->successful()) {
            // Intenta obtener las reuniones
            $meetings = $response->json()['meetings'] ?? [];

            // Si no hay reuniones, muestra un mensaje adecuado
            if (empty($meetings)) {
                return view('docente.zoom.index', ['meetings' => []])->with('error', 'No tienes reuniones programadas.');
            }

            return view('zoom.meetings', compact('meetings'));
        }

        // Si hubo un error al obtener las reuniones
        return redirect()->route('docente.zoom.index')->with('error', 'No se pudieron obtener las reuniones de Zoom');
    }

    /**
     * Crea una nueva reunión.
     */
    public function createMeeting(Request $request)
    {
        // Establecer la zona horaria explícita
        $timezone = 'America/Lima';
        $now = Carbon::now($timezone); // Fecha y hora actual en Lima
        $startTime = Carbon::parse($request->input('start_time'), $timezone); // Fecha elegida por el usuario

        // Validar que no se programen reuniones en el pasado
        if ($startTime->lessThan($now)) {
            return redirect()->back()->with('error', 'No puedes programar una reunión en el pasado. La fecha y hora seleccionada es anterior al momento actual.');
        }

        // Validar horas en el día actual
        if ($startTime->isToday() && $startTime->lessThan($now)) {
            return redirect()->back()->with('error', 'No puedes programar una reunión en una hora pasada para el día actual. Elige una hora futura.');
        }

        // Crear la reunión en Zoom
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return redirect()->route('docente.zoom.index')->with('error', 'No se pudo obtener el token de acceso de Zoom. Intenta nuevamente más tarde.');
        }
        $response = Http::withToken($accessToken)
            ->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic' => $request->input('topic', 'Nueva reunión'),
                'type' => 2,
                'start_time' => $startTime->toIso8601String(),
                'duration' => $request->input('duration', 60),
                'timezone' => $timezone,
            ]);

        if ($response->successful()) {
            return redirect()->route('docente.zoom.index')->with('success', 'Reunión creada exitosamente.');
        }

        // Si Zoom responde con error
        return redirect()->back()->with('error', 'Hubo un problema al crear la reunión en Zoom. Intenta nuevamente.');
    }

    /**
     * Elimina una reunión por ID.
     */
    public function deleteMeeting($meetingId)
    {
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return redirect()->route('docente.zoom.index')->with('error', 'No se pudo obtener el token de acceso.');
        }

        $response = Http::withToken($accessToken)
            ->delete("https://api.zoom.us/v2/meetings/{$meetingId}");

        if ($response->successful()) {
            // Actualizar base de datos (elimino solo el id_zoom_meeting si existe)
            DB::table('clases_asignaturas')
                ->where('id_zoom_meeting', $meetingId) // Buscar por id_zoom_meeting
                ->update(['id_zoom_meeting' => null]); // Eliminar solo el id_zoom_meeting

            return redirect()->route('docente.zoom.index')->with(['success', 'Reunión eliminada correctamente.']);
        }

        return redirect()->back()->with('error', 'No se pudo eliminar la reunión.');
    }

    /**
     * Redirección según perfil del usuario a la página de gestión de reuniones
     */
    public function showMeetingsPage(Request $request)
    {
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return view('docente.zoom.index', ['meetings' => []])
                ->with('error', 'No se pudo obtener el token de acceso.');
        }

        $response = Http::withToken($accessToken)
            ->get('https://api.zoom.us/v2/users/me/meetings');

        if ($response->successful()) {
            $meetings = $response->json()['meetings'] ?? [];
            return view('docente.zoom.index', compact('meetings'));
        }

        return view('docente.zoom.index', ['meetings' => []])
            ->with('error', 'No se pudieron obtener las reuniones de Zoom.');
    }

    /**
     * Listrar reuniones filtrando
     */
    public function getFilteredMeetings(Request $request)
    {
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 500);
        }

        $response = Http::withToken($accessToken)->get('https://api.zoom.us/v2/users/me/meetings');

        if ($response->successful()) {
            $meetings = $response->json()['meetings'] ?? [];
            $now = Carbon::now();

            $filteredMeetings = collect($meetings)->filter(function ($meeting) use ($now, $request) {
                $startTime = Carbon::parse($meeting['start_time']);
                return $request->type === 'upcoming'
                    ? $startTime->isAfter($now)
                    : $startTime->isBefore($now);
            });

            return response()->json($filteredMeetings->values());
        }

        return response()->json(['error' => 'No se pudieron obtener las reuniones de Zoom.'], 500);
    }

    /*----------------------------------------------------------------------------------------
     ********************** MÉTODOS DE GESTIÓN DE REUNIÓN PARA DOCENTES **********************
     *************************** PESTAÑA DE ASISTENCIAS DE ALUMNOS **************************
     *---------------------------------------------------------------------------------------*/

    public function deleteMeetingAndDatabase($meetingId, Request $request)
    {
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 401);
        }

        // Eliminar reunión en Zoom
        $response = Http::withToken($accessToken)
            ->delete("https://api.zoom.us/v2/meetings/{$meetingId}");

        if ($response->successful()) {
            // Actualizar base de datos (elimino solo el id_zoom_meeting)
            DB::table('clases_asignaturas')
                ->where('id', $request->input('idClass'))
                ->update(['id_zoom_meeting' => null]);

            return response()->json(['success' => 'Reunión eliminada correctamente.']);
        }

        return response()->json(['error' => 'No se pudo eliminar la reunión.'], 500);
    }

    public function getLinkMeetingZoom(Request $request, $id_zoom_meeting)
    {
        // Obtiene el token de acceso
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 401);
        }

        // Realiza la solicitud para obtener los detalles de la reunión
        $response = Http::withToken($accessToken)
            ->get("https://api.zoom.us/v2/meetings/$id_zoom_meeting");

        // Verifica si la respuesta fue exitosa
        if ($response->successful()) {
            $meeting = $response->json();

            // Verifica si la reunión tiene un join_url
            if (isset($meeting['join_url'])) {
                return response()->json(['join_url' => $meeting['join_url']]);
            } else {
                return response()->json(['error' => 'No se pudo obtener el link de la reunión.'], 404);
            }
        }

        // Si hubo un error al obtener la reunión
        return response()->json(['error' => 'No se encontró la reunión de Zoom'], 404);
    }

    public function createZoomMeetingForClass(Request $request)
    {
        $validatedData = $request->validate([
            'topic' => 'required|string',
            'start_time' => 'required|date',
            'duration' => 'required|integer|min:1',
            'agenda' => 'nullable|string',
            'idclass' => 'required|integer|exists:clases_asignaturas,id'
        ]);

        $timezone = 'America/Lima';
        $startTime = Carbon::parse($validatedData['start_time'], $timezone)->toIso8601String();

        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return response()->json(['error' => 'Error al obtener el token de acceso.']);
        };

        $response = Http::withToken($accessToken)
            ->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic' => $validatedData['topic'],
                'type' => 2,
                'start_time' => $startTime,
                'duration' => $validatedData['duration'],
                'timezone' => $timezone,
                'agenda' => $validatedData['agenda'],
            ]);

        if ($response->successful()) {
            $meeting = $response->json();
            DB::table('clases_asignaturas')
                ->where('id', $validatedData['idclass'])
                ->update(['id_zoom_meeting' => $meeting['id']]);

            // Convertir toda la respuesta a JSON
            return response()->json([
                'success' => 'Reunión creada exitosamente.',
                'meeting_data' => json_encode($meeting, JSON_PRETTY_PRINT),
                'meeting_start_time' => $validatedData['start_time'],
                'start_time' => $validatedData['start_time'],
            ]);
        }

        return response()->json(['error' => 'Error al crear la reunión en Zoom.']);
    }

    /*----------------------------------------------------------------------------------------
     ********************** MÉTODOS DE GESTIÓN DE REUNIÓN PARA ALUMNOS **********************
     *************************** PESTAÑA DE ASISTENCIAS DE ALUMNOS **************************
     *---------------------------------------------------------------------------------------*/
    private function getAccessToken_InAlumno($id_account, $id_client, $id_client_secret)
    {
        $url = 'https://zoom.us/oauth/token';

        if (is_null($id_account) || is_null($id_client) || is_null($id_client_secret)) {
            return null;  // Devolver null si faltan credenciales
        }

        // Crear la cabecera de autorización Basic Auth
        $authorization = base64_encode("{$id_client}:{$id_client_secret}");

        // Enviar la solicitud a Zoom
        $response = Http::withHeaders([
            'Authorization' => "Basic $authorization",
        ])->asForm()->post($url, [
            'grant_type' => 'account_credentials',
            'account_id' => $id_account,
        ]);

        // Verificar si la respuesta es exitosa y devolver el token
        if ($response->successful()) {
            return $response->json()['access_token'];
        }

        return null;  // Si no se obtiene el token, devolver null
    }

    // Controlador en ZoomController
    public function getLinkMeetingZoom_Alumnos(Request $request)
    {
        // Obtener los parámetros desde el cuerpo de la solicitud
        $params = $request->all();

        // Verificar que los parámetros necesarios estén presentes
        if (!isset($params['id_zoom_meeting'], $params['id_account'], $params['id_client'], $params['id_client_secret'])) {
            return response()->json(['error' => 'Parámetros inválidos.'], 400);
        }

        // Asignar las variables a partir del array de parámetros
        $id_zoom_meeting = $params['id_zoom_meeting'];
        $id_account = $params['id_account'];
        $id_client = $params['id_client'];
        $id_client_secret = $params['id_client_secret'];

        // Obtener el token de acceso
        $accessToken = $this->getAccessToken_InAlumno($id_account, $id_client, $id_client_secret);

        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 401);
        }

        // Realiza la solicitud para obtener los detalles de la reunión
        $response = Http::withToken($accessToken)->get("https://api.zoom.us/v2/meetings/$id_zoom_meeting");

        // Verificar si la respuesta de Zoom es exitosa
        if ($response->successful()) {
            $meeting = $response->json();

            // Verificar si el link de la reunión está presente
            if (isset($meeting['join_url'])) {
                return response()->json(['join_url' => $meeting['join_url']]);
            } else {
                return response()->json(['error' => 'No se pudo obtener el link de la reunión.'], 404);
            }
        } else {
            $errorResponse = $response->json();
            return response()->json([
                'error' => $errorResponse['message'] ?? 'Error desconocido al acceder a la reunión de Zoom.',
                'details' => $errorResponse
            ], $response->status());
        }
    }
}
