<?php

namespace App\Http\Controllers\docente;

use App\Http\Controllers\Controller;
use App\Models\Subjectclass;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DocenteSemanasController extends Controller
{
    public function index()
    {
        return view('docente.semanas.index');
    }

    /**
     * Agregar una nueva semana (clase) al curso seleccionado.
     *
     * Espera en la request:
     * - subject_id (ID de la asignatura / curso)  [obligatorio]
     * - fecha_inicio (YYYY-mm-dd)                [opcional, solo si no hay semanas previas]
     */
    public function store(Request $request)
    {
        if (!$request->ajax()) {
            return response()->json([
                'status' => false,
                'mensaje' => 'Petición no válida'
            ], 400);
        }

        // Validación básica
        $request->validate([
            'subject_id' => 'required|integer|exists:asignaturas,id',
            'fecha_inicio' => 'nullable|date',
        ]);

        $subjectId   = (int) $request->input('subject_id');
        $fechaInicio = $request->input('fecha_inicio'); // puede venir null

        // 1. Obtener los horarios del curso seleccionado
        $horarios = DB::table('horarios as h')
            ->select('h.id')
            ->where('h.subject_id', $subjectId)
            ->get();

        if ($horarios->isEmpty()) {
            return response()->json(['status'  => false, 'mensaje' => 'El curso seleccionado no tiene horarios registrados.']);
        }

        $idsHorarios = $horarios->pluck('id')->toArray();

        // 2. Obtener todas las clases existentes de esos horarios (para calcular el número de semana)
        $clasesExistentes = DB::table('clases_asignaturas as ca')
            ->select('ca.id', 'ca.schedule_id', 'ca.nombre', 'ca.fecha')
            ->whereIn('ca.schedule_id', $idsHorarios)
            ->where('ca.nombre', 'LIKE', 'Semana %')
            ->get();

        // 3. Calcular el siguiente número de semana: Semana N+1
        $maxSemana = 0;

        foreach ($clasesExistentes as $clase) {
            if (preg_match('/Semana\s+(\d+)/i', $clase->nombre, $coincidencia)) {
                $num = (int) $coincidencia[1];
                if ($num > $maxSemana) {
                    $maxSemana = $num;
                }
            }
        }

        $numeroNuevaSemana = $maxSemana + 1;

        // 4. Si no hay ninguna semana previa y no mandan fecha_inicio, no sabemos desde cuándo generar
        if ($clasesExistentes->isEmpty() && !$fechaInicio) {
            return response()->json([
                'status'  => false,
                'mensaje' => 'No existen semanas previas y no se envió fecha_inicio.'
            ]);
        }

        DB::beginTransaction();

        try {
            foreach ($horarios as $horario) {
                // Buscar la última clase por horario para calcular la nueva fecha
                $ultimaClaseHorario = $clasesExistentes
                    ->where('schedule_id', $horario->id)
                    ->sortByDesc('fecha')
                    ->first();

                if ($ultimaClaseHorario) {
                    // Si ya hay clases, la nueva semana es +7 días
                    $baseDate = $ultimaClaseHorario->fecha;
                    $nuevaFecha = date('Y-m-d', strtotime($baseDate . ' + 7 days'));
                } else {
                    // Si no hay ninguna clase para ese horario, usamos fecha_inicio
                    $nuevaFecha = $fechaInicio;
                }

                $item = new Subjectclass();
                $item->schedule_id = $horario->id;
                $item->fecha       = $nuevaFecha;
                $item->nombre      = 'Semana ' . $numeroNuevaSemana;
                $item->estado      = 0; // pendiente o activo por defecto según tu lógica

                $item->save();
            }

            DB::commit();

            return response()->json([
                'status'  => true,
                'mensaje' => 'Se ha creado la Semana ' . $numeroNuevaSemana . ' para el curso seleccionado.'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'status'  => false,
                'mensaje' => 'Ocurrió un error al generar la nueva semana.'
            ], 500);
        }
    }
}
