<?php

namespace App\Http\Controllers\Alumno;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class HomeworkController extends Controller
{
    public function index()
    {
        return view('alumno.tareas.index');
    }

    // 1. OBTENER PERIODOS (Corregido)
    public function getPeriods(Request $request)
    {
        if ($request->ajax()) {
            $userId = Auth::id();

            $semestres = DB::table('semestres as s')
                ->join('matriculas_semestres as ms', 'ms.semester_id', '=', 's.id')
                ->join('estudiantes as e', 'e.id', '=', 'ms.student_id')
                ->where('e.user_id', $userId)
                ->select(
                    's.id',
                    's.numero',
                    's.anho'
                )
                ->distinct()
                ->orderBy('s.anho', 'desc')
                ->orderBy('s.numero', 'desc')
                ->get();

            return response()->json($semestres);
        }

        return response()->json(['error' => 'Acceso denegado'], 403);
    }

    public function getCourses(Request $request, $idsemester)
    {
        if ($request->ajax()) {
            $userId = Auth::id();

            $data = DB::table('asignaturas as a')
                ->select(
                    'a.id as asignatura_id',
                    'cu.nombre as curso_nombre'
                )
                ->join('cursos as cu', 'a.course_id', '=', 'cu.id')
                ->join('matriculas_asignaturas as ma', 'ma.subject_id', '=', 'a.id')
                ->join('estudiantes as e', 'e.id', '=', 'ma.student_id')
                ->join('matriculas_semestres as ms', function ($join) {
                    $join->on('ms.student_id', '=', 'e.id')
                        ->whereColumn('ms.semester_id', '=', 'a.semester_id');
                })
                ->where('e.user_id', $userId)
                ->where('a.semester_id', $idsemester)
                ->orderBy('cu.nombre')
                ->get();

            return response()->json($data);
        }

        return response()->json(['error' => 'Acceso denegado'], 403);
    }

    public function getTasks(Request $request, $asignatura_id)
    {
        if ($request->ajax()) {
            $userId = Auth::id();

            $student = DB::table('estudiantes')->where('user_id', $userId)->first();
            if (! $student) {
                return response()->json([]);
            }

            $matricula = DB::table('matriculas_asignaturas')
                ->where('student_id', $student->id)
                ->where('subject_id', $asignatura_id)
                ->first();
            $enrollmentId = $matricula ? $matricula->id : null;
            $actividades = DB::table('actividades as ac')
                ->join('indicadores as i', 'i.id', '=', 'ac.indicator_id')
                ->where('i.subject_id', $asignatura_id)
                ->leftJoin('notas as n', 'n.activity_id', '=', 'ac.id')
                ->leftJoin('notas_estudiantes as ne', function ($join) use ($enrollmentId) {
                    $join->on('ne.grade_id', '=', 'n.id')
                        ->where('ne.subjectenrollment_id', '=', $enrollmentId);
                })
                ->where('n.tipo_nota', 1)
                ->select(
                    'ac.id as actividad_id',
                    'ac.nombre as actividad_nombre',
                    'i.nombre as indicador_nombre',
                    'n.nombre as nombre_nota',
                    'n.id as id_definicion_nota',
                    DB::raw('IFNULL(n.fecha_limite, ac.fecha) as fecha_entrega'),
                    'n.porcentaje',
                    'ne.nota as nota_obtenida',
                    'ne.documento',
                    'ne.asistencia'
                )
                ->orderBy('i.id', 'asc')
                ->orderBy('n.fecha_limite', 'asc')
                ->get();

            return response()->json($actividades);
        }

        return response()->json(['error' => 'Acceso denegado'], 403);
    }

    public function uploadHomework(Request $request)
    {
        $validator = \Illuminate\Support\Facades\Validator::make($request->all(), [
            'file_tarea' => 'required|file',
            'asignatura_id' => 'required',
            'nota_id' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'message' => $validator->errors()->first()], 422);
        }

        try {
            DB::beginTransaction();

            $userId = Auth::id();
            $student = DB::table('estudiantes')->where('user_id', $userId)->first();

            // 1. OBTENCIÓN DE DATOS
            $cursoData = DB::table('asignaturas as a')
                ->join('cursos as c', 'c.id', '=', 'a.course_id')
                ->where('a.id', $request->asignatura_id)
                ->select('c.nombre', 'c.id as curso_id')
                ->first();

            $notaData = DB::table('notas')->where('id', $request->nota_id)->first();

            $matricula = DB::table('matriculas_asignaturas')
                ->where('student_id', $student->id)
                ->where('subject_id', $request->asignatura_id)
                ->first();

            if (! $student || ! $cursoData || ! $notaData || ! $matricula) {
                return response()->json(['status' => false, 'message' => 'Datos no encontrados.']);
            }

            // 2. BUSCAR O CREAR EL REGISTRO PARA OBTENER EL ID
            $notaEstudiante = DB::table('notas_estudiantes')
                ->where('grade_id', $request->nota_id)
                ->where('subjectenrollment_id', $matricula->id)
                ->first();

            $idNotaEstudiante = null;

            if ($notaEstudiante) {
                $idNotaEstudiante = $notaEstudiante->id;
            } else {
                $idNotaEstudiante = DB::table('notas_estudiantes')->insertGetId([
                    'grade_id' => $request->nota_id,
                    'subjectenrollment_id' => $matricula->id,
                    'created_at' => now(),
                    'updated_at' => now(),
                    'documento' => null,
                ]);
            }

            // 3. DEFINIR RUTAS DE CARPETAS
            $studentFolder = "students/{$student->id}";
            $courseNameClean = Str::slug($cursoData->nombre, '_')."_{$cursoData->curso_id}";
            $noteNameClean = Str::slug($notaData->nombre, '_')."_{$notaData->id}";

            // Ruta relativa (sin 'public')
            $relativePath = "{$studentFolder}/{$courseNameClean}/{$noteNameClean}";

            // 4. GUARDAR EL ARCHIVO CON EL ID COMO NOMBRE
            $file = $request->file('file_tarea');
            $extension = $file->getClientOriginalExtension();
            $fileName = $idNotaEstudiante.'.'.$extension;
            $pathSaved = $file->storeAs($relativePath, $fileName, 'public');

            if (! $pathSaved) {
                throw new \Exception('No se pudo guardar el archivo.');
            }
            // 5. ACTUALIZAR LA RUTA EN LA BASE DE DATOS
            DB::table('notas_estudiantes')
                ->where('id', $idNotaEstudiante)
                ->update([
                    'documento' => $pathSaved,
                    'updated_at' => now(),
                ]);

            DB::commit();

            return response()->json(['status' => true, 'message' => 'Tarea subida correctamente']);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json(['status' => false, 'message' => 'Error interno: '.$e->getMessage()]);
        }
    }
}