<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Http\Controllers\sidebar\SidebarController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class TransfersController extends Controller
{
    public function index(Request $request)
    {
        return view('admin.estudiantes.traslados.index');
    }

    public function searchStudent(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nroidenti' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Número de documento inválido.'], 400);
        }

        try {
            $studentData = DB::table('usuarios as u')
                ->join('estudiantes as e', 'u.id', '=', 'e.user_id')
                ->select(
                    'e.id as student_id',
                    'u.nombres',
                    'u.apellido_pa',
                    'u.apellido_ma'
                )
                ->where('u.nroidenti', $request->nroidenti)
                ->first();

            if ($studentData) {
                return response()->json($studentData);
            } else {
                return response()->json(['error' => 'Estudiante no encontrado.'], 404);
            }
        } catch (\Exception $e) {
            Log::error('Error al buscar estudiante: '.$e->getMessage());

            return response()->json(['error' => 'Error interno del servidor.'], 500);
        }
    }

    public function list(Request $request)
    {
        $traslados = DB::table('traslados as t')
            ->join('estudiantes as e', 't.student_id', '=', 'e.id')
            ->join('usuarios as u', 'e.user_id', '=', 'u.id')
            ->leftJoin('instituciones as io', 't.institucion_origen', '=', 'io.id')
            ->leftJoin('instituciones as id', 't.institucion_destino', '=', 'id.id')
            ->select(
                't.id',
                'u.nroidenti',
                DB::raw("CONCAT(u.nombres, ' ', u.apellido_pa, ' ', u.apellido_ma) as nombre_completo"),
                't.tipo',
                't.fecha',
                'io.name as institucion_origen_nombre',
                't.carrera_origen',
                'id.name as institucion_destino_nombre',
                't.carrera_destino',
                't.documento'
            )
            ->get();

        return response()->json(['data' => $traslados]);
    }

    public function verDocumento($file)
    {
        $file = str_replace(['../', '..\\'], '', $file);
        $file = ltrim($file, '/');

        $path = storage_path('app/public/'.$file);

        if (! file_exists($path)) {
            abort(404, 'Documento no encontrado.');
        }

        $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        $mimeMap = [
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
        ];

        $mime = $mimeMap[$ext] ?? 'application/octet-stream';

        if (in_array($ext, ['pdf', 'jpg', 'jpeg', 'png'])) {

            $headers = [
                'Content-Type' => $mime,
                'Content-Disposition' => 'inline; filename="'.basename($file).'"',
            ];

            // Devolvemos el archivo con las cabeceras
            return response()->file($path, $headers);
        }
    }

    public function getLocalInstitution(Request $request)
    {
        try {
            $myInstitution = DB::table('info_institucional')
                ->select('id', 'nombre as name')
                ->first();

            if ($myInstitution) {
                return response()->json($myInstitution);
            } else {
                return response()->json(['error' => 'Institución local no configurada.'], 404);
            }
        } catch (\Exception $e) {
            Log::error('Error al cargar institución local: '.$e->getMessage());

            return response()->json(['error' => 'Error interno del servidor.'], 500);
        }
    }

    public function getExternalInstitutions(Request $request)
    {
        try {
            $externalInstitutions = DB::table('instituciones')
                ->select('id', 'name')
                ->orderBy('name', 'asc')
                ->get();

            return response()->json($externalInstitutions);

        } catch (\Exception $e) {
            Log::error('Error al cargar instituciones externas: '.$e->getMessage());

            return response()->json(['error' => 'No se pudieron cargar las instituciones'], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'student_id' => 'required|integer|exists:estudiantes,id',
            'tipo' => 'required|string|in:Ingreso,Salida',
            'fecha' => 'required|date',
            'institucion_origen' => 'required_if:tipo,Ingreso|nullable|integer',
            'carrera_origen' => 'required|string|max:255',
            'institucion_destino' => 'required_if:tipo,Salida|nullable|integer',
            'carrera_destino' => 'required|string|max:255',
            'archivo' => 'nullable|file|mimes:pdf,doc,docx,jpg,png',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        DB::beginTransaction();
        try {
            $filePath = null;

            if ($request->hasFile('archivo')) {
                $file = $request->file('archivo');
                $fileName = $file->getClientOriginalName();

                $studentId = $request->student_id;
                $directory = "students/{$studentId}/traslados";

                $filePath = $file->storeAs($directory, $fileName, 'public');
            }

            $myInstitution = DB::table('info_institucional')->select('id')->first();
            if (! $myInstitution) {
                throw new \Exception('Configuración de institución local no encontrada.');
            }
            $myInstitutionId = $myInstitution->id;
            $tipoInt = ($request->tipo == 'Ingreso') ? 0 : 1;

            $dataToInsert = [
                'student_id' => $request->student_id,
                'tipo' => $tipoInt,
                'fecha' => $request->fecha,
                'carrera_origen' => $request->carrera_origen,
                'carrera_destino' => $request->carrera_destino,
                'documento' => $filePath,
                'created_at' => now(),
                'updated_at' => now(),
            ];

            if ($request->tipo == 'Ingreso') {
                $dataToInsert['institucion_origen'] = $request->institucion_origen;
                $dataToInsert['institucion_destino'] = $myInstitutionId;
            } else {
                $dataToInsert['institucion_origen'] = $myInstitutionId;
                $dataToInsert['institucion_destino'] = $request->institucion_destino;
            }
            DB::table('traslados')->insert($dataToInsert);
            DB::commit();

            return response()->json(['success' => 'Traslado registrado exitosamente.']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al guardar traslado: '.$e->getMessage());
            if (isset($filePath) && Storage::disk('public')->exists($filePath)) {
                Storage::disk('public')->delete($filePath);
            }

            return response()->json(['error' => 'No se pudo registrar el traslado. Error: '.$e->getMessage()], 500);
        }
    }

    public function show($id)
    {
        $traslado = DB::table('traslados as t')
            ->join('estudiantes as e', 't.student_id', '=', 'e.id')
            ->join('usuarios as u', 'e.user_id', '=', 'u.id')
            ->select(
                't.*',
                'u.nroidenti',
                DB::raw("CONCAT(u.nombres, ' ', u.apellido_pa, ' ', u.apellido_ma) as nombre_completo")
            )
            ->where('t.id', $id)
            ->first();

        if ($traslado) {
            $traslado->tipo = $traslado->tipo == 0 ? 'Ingreso' : 'Salida';

            return response()->json($traslado);
        }

        return response()->json(['error' => 'Registro no encontrado'], 404);
    }

    public function update(Request $request, $id)
    {
        // 1. El validador está bien
        $validator = Validator::make($request->all(), [
            'student_id' => 'required|integer|exists:estudiantes,id',
            'tipo' => 'required|string|in:Ingreso,Salida',
            'fecha' => 'required|date',
            'institucion_origen' => 'required_if:tipo,Ingreso|nullable|integer',
            'carrera_origen' => 'required|string|max:255',
            'institucion_destino' => 'required_if:tipo,Salida|nullable|integer',
            'carrera_destino' => 'required|string|max:255',
            'archivo' => 'nullable|file|mimes:pdf,doc,docx,jpg,png',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // 2. Buscamos el registro antiguo
        $oldTraslado = DB::table('traslados')->where('id', $id)->first();
        if (! $oldTraslado) {
            return response()->json(['error' => 'Registro no encontrado.'], 404);
        }

        DB::beginTransaction();
        try {
            $filePath = $oldTraslado->documento;

            if ($request->hasFile('archivo')) {

                if ($oldTraslado->documento && Storage::disk('public')->exists($oldTraslado->documento)) {
                    Storage::disk('public')->delete($oldTraslado->documento);
                }

                $file = $request->file('archivo');
                $fileName = $file->getClientOriginalName();

                $studentId = $request->student_id;
                $directory = "students/{$studentId}/traslados";

                $filePath = $file->storeAs($directory, $fileName, 'public');
            }

            $myInstitution = DB::table('info_institucional')->select('id')->first();
            if (! $myInstitution) {
                throw new \Exception('Configuración de institución local no encontrada.');
            }
            $myInstitutionId = $myInstitution->id;

            $tipoInt = ($request->tipo == 'Ingreso') ? 0 : 1;

            $dataToUpdate = [
                'student_id' => $request->student_id,
                'tipo' => $tipoInt,
                'fecha' => $request->fecha,
                'carrera_origen' => $request->carrera_origen,
                'carrera_destino' => $request->carrera_destino,
                'documento' => $filePath,
                'updated_at' => now(),
            ];

            if ($request->tipo == 'Ingreso') {
                $dataToUpdate['institucion_origen'] = $request->institucion_origen;
                $dataToUpdate['institucion_destino'] = $myInstitutionId;
            } else {
                $dataToUpdate['institucion_origen'] = $myInstitutionId;
                $dataToUpdate['institucion_destino'] = $request->institucion_destino;
            }

            DB::table('traslados')->where('id', $id)->update($dataToUpdate);

            DB::commit();

            return response()->json(['success' => 'Traslado actualizado exitosamente.']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al actualizar traslado: '.$e->getMessage());

            return response()->json(['error' => 'No se pudo actualizar el traslado. Error: '.$e->getMessage()], 500);
        }
    }

    public function destroy($id)
    {
        $traslado = DB::table('traslados')->where('id', $id)->first();

        if (! $traslado) {
            return response()->json(['error' => 'Registro no encontrado.'], 404);
        }

        DB::beginTransaction();
        try {
            // 1. Borrar el archivo del storage (si existe)
            if ($traslado->documento && Storage::disk('public')->exists($traslado->documento)) {
                Storage::disk('public')->delete($traslado->documento);
            }

            // 2. Borrar el registro de la BD
            DB::table('traslados')->where('id', $id)->delete();

            DB::commit();

            return response()->json(['success' => 'Traslado eliminado exitosamente.']);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al eliminar traslado: '.$e->getMessage());

            return response()->json(['error' => 'No se pudo eliminar el registro.'], 500);
        }
    }
}
