<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\TeacherDocument;
use App\Models\TeacherRequirement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class TeacherDocumentController extends Controller
{
    public function list(Request $request)
    {
        if ($request->ajax()) {
            $validator = Validator::make($request->all(), [
                'teacher_id' => 'required|integer|exists:docentes,id',
            ]);

            if ($validator->fails()) {
                return response()->json(["status" => false, "mensaje" => $validator->errors()->first()]);
            }

            $teacherId = $request->input('teacher_id');

            // Traer todos los requisitos con el documento del docente
            $requirements = TeacherRequirement::with([
                'teachersDocuments' => function ($q) use ($teacherId) {
                    $q->where('teacher_id', $teacherId)
                        ->with(['administrator.user']); // para approved_by_name
                }
            ])->get();

            // Mapear a una colección "plana" para DataTables
            $data = $requirements->map(function ($req) use ($teacherId) {
                // Tomamos el primer documento del docente para ese requisito (si hay)
                $doc = $req->teachersDocuments->first();

                return [
                    'teacher_requirement_id' => $req->id,
                    'requirement_name' => $req->nombre,
                    'id' => $doc->id ?? null,
                    'documento' => $doc->documento ?? null,
                    'estado' => $doc->estado ?? null,
                    'approved_by' => $doc->approved_by ?? null,
                    'approved_by_name' => ($doc && $doc->administrator && $doc->administrator->user)
                        ? ($doc->administrator->user->nombres . ' ' . $doc->administrator->user->apellido_pa)
                        : '--',
                ];
            });

            return \Yajra\DataTables\DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return  response()->json(["status" => false, "mensaje" => 'Error: no se pueden cargar los archivos']);
    }


    public function update(Request $request, $id)
    {
        $item = TeacherDocument::find($id);

        if (!$item) {
            return response()->json(["status" => false, "mensaje" => 'Registro no encontrado']);
        }

        $validator = Validator::make($request->all(), [
            'txt-estado' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return response()->json(["status" => false, "mensaje" => $validator->errors()->first()]);
        }

        try {
            $item->estado = $request->input('txt-estado');
            $item->approved_by = auth()->user()->id;

            $item->save();

            return response()->json(["status" => true, "mensaje" => 'Registro actualizado con éxito']);
        } catch (\Exception $e) {
            return response()->json(["status" => false, "mensaje" => 'Error al actualizar el registro.']);
        }
    }


    public function destroy($id)
    {
        $item = TeacherDocument::find($id);

        if (!$item) {
            return response()->json(["status" => false, "mensaje" => 'El registro no existe']);
        }

        if ($item->documento) {
            $ruta = str_replace('public/', '', $item->documento);
            Storage::disk('public')->delete($ruta);
        }

        if ($item->delete()) {
            return response()->json(["status" => true, "mensaje" => 'El registro fue eliminado']);
        } else {
            return response()->json(["status" => false, "mensaje" => 'El registro no fue eliminado']);
        }
    }


    public function verDocumentoRequisito($id)
    {
        $item = TeacherDocument::find($id);

        if ($item->documento == "--" || $item->documento == "") {
            $file = storage_path('app/public/sinImagen.jpg');
            return $this->viewFile($file);
        }

        $file = storage_path('app/') . $item->documento;

        if (file_exists($file)) {
            return $this->viewFile($file);
        } else {
            $file = storage_path('app/public/sinImagen.jpg');
            return $this->viewFile($file);
        }
    }

    public function viewFile($file)
    {
        $extension = pathinfo($file, PATHINFO_EXTENSION);

        // Verificar la extensión y establecer el tipo de contenido correspondiente
        switch ($extension) {
            case 'pdf':
                $contentType = 'application/pdf';
                break;
            case 'png':
                $contentType = 'image/png';
                break;
            case 'jpg':
            case 'jpeg':
                $contentType = 'image/jpeg';
                break;
            default:
                abort(404, 'Formato de archivo no admitido.');
        }

        $headers = [
            'Content-Type' => $contentType,
        ];

        $fileTitle = 'documento.' . $extension;
        return response()->download($file, $fileTitle, $headers, 'inline');
    }
}
