<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\ItemsTitulacion;
use App\Models\DocumentoTitulacion;   
use App\Models\Student;             
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class StudentItemsTitulacionController extends Controller
{
    public function index(Request $request)
    {
        return view('admin.seguimiento_egresado.items_titulacion.index');
    }

    public function list(Request $request)
    {
        if ($request->ajax()) {
            $data = ItemsTitulacion::select(
                'id',
                'concepto',
                'descripcion',
                'estado',
                'created_at',
                'updated_at'
            )->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return response()->json([
            "status"  => false,
            "mensaje" => 'No se puede cargar la información'
        ]);
    }

    /**
     * STORE:
     *  - Crea el item.
     *  - Crea automáticamente registros en documento_titulacion
     *    para TODOS los estudiantes egresados (estudiantes.estado = 2).
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'txt-concepto'    => 'required|string|max:255',
            'txt-descripcion' => 'nullable|string',
            'estado'          => 'nullable|in:0,1',
        ], [
            'txt-concepto.required' => 'El concepto es obligatorio.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error de validación',
                "errors"  => $validator->errors()
            ]);
        }

        try {
            DB::beginTransaction();

            // Normalizamos el estado del ÍTEM:
            // - Si no viene => 1 (Activo)
            // - Si viene 0 o 1 => se respeta
            $estadoItem = $request->filled('estado')
                ? (int) $request->input('estado')
                : 1;

            $item = new ItemsTitulacion();
            $item->concepto    = $request->input('txt-concepto');
            $item->descripcion = $request->input('txt-descripcion');
            $item->estado      = $estadoItem;
            $item->save();

            // VISIBILIDAD de los documentos según el ítem:
            // - Ítem Inactivo (0) => visible = 0
            // - Ítem Activo   (1) => visible = 1
            $visibleDocumento = $estadoItem == 1 ? 1 : 0;

            // Obtener todos los estudiantes egresados (estado = 2, según tu lógica)
            $egresadosIds = Student::where('estado', 2)
                ->pluck('id')
                ->toArray();

            if (!empty($egresadosIds)) {
                $now  = now();
                $rows = [];

                foreach ($egresadosIds as $studentId) {
                    $rows[] = [
                        'student_id'         => $studentId,
                        'item_titulacion_id' => $item->id,
                        'archivo'            => null,
                        'estado'             => 0, // SIEMPRE pendiente al crear
                        'permiso_observacion'=> 0,
                        'es_visible'            => $visibleDocumento,
                        'created_at'         => $now,
                        'updated_at'         => $now,
                    ];
                }

                DocumentoTitulacion::insert($rows);
            }

            DB::commit();

            return response()->json([
                "status"  => true,
                "mensaje" => 'Éxito: ítem y documentos asociados registrados correctamente.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al insertar el registro.',
                "error"   => $e->getMessage()
            ]);
        }
    }




    /**
     * UPDATE:
     *  - Actualiza concepto / descripción.
     */
    public function update(Request $request, $id)
    {
        $item = ItemsTitulacion::find($id);
    
        if (!$item) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Registro no encontrado'
            ], 404);
        }
    
        $validator = Validator::make($request->all(), [
            'txt-concepto'    => 'required|string|max:255',
            'txt-descripcion' => 'nullable|string',
        ], [
            'txt-concepto.required' => 'El concepto es obligatorio.',
        ]);
    
        if ($validator->fails()) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error de validación',
                "errors"  => $validator->errors()
            ]);
        }
    
        try {
            DB::beginTransaction();
    
            $item->concepto    = $request->input('txt-concepto');
            $item->descripcion = $request->input('txt-descripcion');
            // NO tocamos estado aquí
            $item->save();
    
            DB::commit();
    
            return response()->json([
                "status"  => true,
                "mensaje" => 'Éxito: registro actualizado correctamente'
            ]);
    
        } catch (\Exception $e) {
            DB::rollBack();
    
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al actualizar el registro',
                "error"   => $e->getMessage()
            ]);
        }
    }
    
    

    /**
     * DESTROY:
     *  - Solo permite eliminar si NO existen documentos de titulación
     *    asociados al item.
     */
    public function destroy($id)
    {
        $item = ItemsTitulacion::find($id);

        if (!$item) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Registro no encontrado'
            ], 404);
        }

        // Verificar si existen documentos asociados
        $documentosCount = DocumentoTitulacion::where('item_titulacion_id', $item->id)->count();

        if ($documentosCount > 0) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'No se puede eliminar: existen documentos de titulación asociados a este ítem.'
            ]);
        }

        try {
            if ($item->delete()) {
                return response()->json([
                    "status"  => true,
                    "mensaje" => 'Éxito: registro eliminado correctamente'
                ]);
            }

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error: el registro no pudo ser eliminado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al eliminar el registro',
                "error"   => $e->getMessage()
            ]);
        }
    }

    /**
     * UPDATEESTADO:
     *  - Cambia solo el estado del ítem (0 = Inactivo, 1 = Activo).
     *  - No modifica el estado (0,1,2,3) de los documentos.
     *  - Solo sincroniza el campo "visible" de documento_titulacion:
     *      Ítem estado = 0  → es_visible = 0
     *      Ítem estado = 1  → es_visible = 1
     */
    public function updateestado(Request $request, $id)
    {
        $item = ItemsTitulacion::find($id);

        if (!$item) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Registro no encontrado'
            ], 404);
        }

        if (!$request->has('estado')) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Debe enviar el parámetro estado (0 o 1).'
            ]);
        }

        $estado = $request->input('estado');

        if (!in_array($estado, [0, 1, '0', '1'], true)) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Valor de estado inválido. Solo se acepta 0 o 1.'
            ]);
        }

        try {
            DB::beginTransaction();

            $item->estado = (int) $estado;
            $item->save();

            // Sincronizar "visible" en documento_titulacion
            $es_visible = $item->estado == 1 ? 1 : 0;

            DocumentoTitulacion::where('item_titulacion_id', $item->id)
                ->update(['es_visible' => $es_visible]);

            DB::commit();

            return response()->json([
                "status"  => true,
                "mensaje" => 'Estado del ítem y visibilidad de documentos actualizados correctamente.',
                "estado"  => $item->estado
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al actualizar el estado.',
                "error"   => $e->getMessage()
            ]);
        }
    }

    

}
