<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\DocumentoTitulacion;
use App\Models\ItemsTitulacion;
use App\Models\Student;
use Illuminate\Database\QueryException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class StudentDocumentosTitulacionController extends Controller
{
    /**
     * Muestra la vista principal de gestión de documentos de titulación.
     */
    public function index(Request $request)
    {
        return view('admin.seguimiento_egresado.documentos_titulacion.index');
    }


    /**
     * Lista los documentos de titulación para DataTables.
     * Puedes filtrar opcionalmente por idstudent.
     */
    public function list(Request $request)
    {
        if ($request->ajax()) {
            $idStudent = $request->input('idstudent');

            if (empty($idStudent)) {
                $data = collect([]); // colección vacía
                return DataTables::of($data)
                    ->addIndexColumn()
                    ->make(true);
            }

            $data = DB::table('documento_titulacion as dt')
                ->join('estudiantes as es', 'es.id', '=', 'dt.student_id')
                ->join('usuarios as us', 'us.id', '=', 'es.user_id')
                ->join('items_titulacion as it', 'it.id', '=', 'dt.item_titulacion_id')
                ->select(
                    'dt.id',
                    'dt.student_id',
                    'dt.item_titulacion_id',
                    'dt.archivo',
                    'dt.estado',
                    'dt.permiso_observacion',
                    'dt.es_visible',
                    'dt.created_at',
                    'dt.updated_at',
                    'it.concepto as item_concepto',
                    'it.descripcion as item_descripcion',
                    'it.estado as item_estado',
                    'us.nroidenti',
                    DB::raw("CONCAT(us.apellido_pa, ' ', us.apellido_ma, ', ', us.nombres) AS estudiante")
                )
                ->where('dt.student_id', $idStudent)
                ->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return response()->json([
            "status"  => false,
            "mensaje" => 'No se puede cargar la información'
        ]);
    }



    /*
    Listar Items de Titulación activos
    list_itemstitulacion
    */
    public function list_itemstitulacion(Request $request)
    {
        $itemsTitulacion = ItemsTitulacion::where('estado', 1)->get();

        return response()->json([
            "status"  => true,
            "mensaje" => 'Éxito: itemstitulacion cargados correctamente',
            "itemsTitulacion" => $itemsTitulacion
        ]);
    }


/**
 * Registra un nuevo documento de titulación.
 * Guarda el archivo en storage y la ruta en BD.
 */
public function store(Request $request)
{
    $validator = Validator::make($request->all(), [
        'student_id'          => 'required|exists:estudiantes,id',
        'item_titulacion_id'  => 'required|exists:items_titulacion,id',
        // Estados: 0 pendiente, 1 aprobado, 2 rechazado, 3 observado
        'estado'              => 'required|in:0,1,2,3',
        'permiso_observacion' => 'required|in:0,1',
        'es_visible'          => 'required|in:0,1',
    ], [
        'student_id.required'          => 'El estudiante es obligatorio.',
        'item_titulacion_id.required'  => 'El ítem de titulación es obligatorio.',
        'estado.required'              => 'El estado es obligatorio.',
        'permiso_observacion.required' => 'El permiso de observación es obligatorio.',
        'es_visible.required'          => 'El es_visible es obligatorio.',
    ]);

    if ($validator->fails()) {
        return response()->json([
            "status"  => false,
            "mensaje" => 'Error de validación',
            "errors"  => $validator->errors()
        ]);
    }

    // Validación lógica previa (por si acaso)
    $existe = DocumentoTitulacion::where('student_id', $request->student_id)
        ->where('item_titulacion_id', $request->item_titulacion_id)
        ->exists();

    if ($existe) {
        return response()->json([
            "status"  => false,
            "mensaje" => 'Ya existe un documento de titulación para este estudiante y este ítem.'
        ]);
    }

    DB::beginTransaction();

    try {
        // 1. Crear el registro SIN archivo para obtener el ID
        $doc = new DocumentoTitulacion();
        $doc->student_id          = $request->input('student_id');
        $doc->item_titulacion_id  = $request->input('item_titulacion_id');
        $doc->estado              = $request->input('estado'); // 0–3
        $doc->permiso_observacion = $request->input('permiso_observacion', 0);
        $doc->es_visible          = $request->input('es_visible'); // 0–1
        $doc->archivo             = null;

        if (!$doc->save()) {
            DB::rollBack();
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error: el documento no pudo ser registrado'
            ]);
        }

        // 2. Guardar archivo físico usando el id del documento
        if ($request->hasFile('archivo')) {
            $file      = $request->file('archivo');
            $extension = $file->getClientOriginalExtension();

            $folderName = 'public/seguimiento-egresado/' . $doc->id . '/';
            Storage::disk('local')->makeDirectory($folderName);

            $fileName = 'documento.' . $extension;
            $file->storeAs($folderName, $fileName);

            $filePath   = $folderName . $fileName;
            $doc->archivo = $filePath;
            $doc->save();
        }

        DB::commit();

        return response()->json([
            "status"  => true,
            "mensaje" => 'Éxito: documento registrado correctamente'
        ]);

    } catch (QueryException $e) {
        DB::rollBack();

        // Si falla por la UNIQUE KEY (código 23000 en la mayoría de motores)
        if ($e->getCode() == '23000') {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Ya existe un documento de titulación para este estudiante y este ítem (restricción única).',
                "error"   => $e->getMessage()
            ]);
        }

        return response()->json([
            "status"  => false,
            "mensaje" => 'Error en la base de datos al registrar el documento',
            "error"   => $e->getMessage()
        ]);

    } catch (\Exception $e) {
        DB::rollBack();
        return response()->json([
            "status"  => false,
            "mensaje" => 'Error inesperado al registrar el documento',
            "error"   => $e->getMessage()
        ]);
    }
}


    /**
     * Actualiza un documento de titulación.
     * Si se envía un nuevo archivo, reemplaza el anterior en storage.
     */
    public function update(Request $request, $idstudentdocument)
    {
        $doc = DocumentoTitulacion::find($idstudentdocument);

        if (!$doc) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Documento no encontrado'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            // Ya NO pedimos student_id ni item_titulacion_id
            'archivo'    => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:90000',
            'es_visible' => 'required|in:0,1',
        ], [
            'es_visible.required' => 'El es_visible es obligatorio.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error de validación',
                "errors"  => $validator->errors()
            ]);
        }

        try {
            // Solo actualizamos visibilidad
            $doc->es_visible = $request->input('es_visible');

            // Si viene un nuevo archivo, se reemplaza
            if ($request->hasFile('archivo')) {
                if ($doc->archivo && Storage::disk('local')->exists($doc->archivo)) {
                    Storage::disk('local')->delete($doc->archivo);
                }

                $file      = $request->file('archivo');
                $extension = $file->getClientOriginalExtension();

                $folderName = 'public/seguimiento-egresado/' . $doc->id . '/';
                Storage::disk('local')->makeDirectory($folderName);

                $fileName = 'documento.' . $extension;
                $file->storeAs($folderName, $fileName);

                $filePath = $folderName . $fileName;
                $doc->archivo = $filePath;
            }

            if ($doc->save()) {
                return response()->json([
                    "status"  => true,
                    "mensaje" => 'Éxito: documento actualizado correctamente'
                ]);
            }

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error: el documento no pudo ser actualizado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al actualizar el documento',
                "error"   => $e->getMessage()
            ]);
        }
    }


    /**
     * Elimina un documento de titulación.
     * Borra el registro y el archivo en storage.
     */
    public function destroy($idstudentdocument)
    {
        $doc = DocumentoTitulacion::find($idstudentdocument);

        if (!$doc) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Documento no encontrado'
            ], 404);
        }

        try {
            // Eliminar archivo físico si existe
            if ($doc->archivo && Storage::disk('local')->exists($doc->archivo)) {
                Storage::disk('local')->delete($doc->archivo);
            }

            // Eliminar carpeta completa del documento
            $folderName = 'public/seguimiento-egresado/' . $doc->id . '/';
            if (Storage::disk('local')->exists($folderName)) {
                Storage::disk('local')->deleteDirectory($folderName);
            }

            if ($doc->delete()) {
                return response()->json([
                    "status"  => true,
                    "mensaje" => 'Éxito: documento eliminado correctamente'
                ]);
            }

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error: el documento no pudo ser eliminado'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al eliminar el documento',
                "error"   => $e->getMessage()
            ]);
        }
    }

    /**
     * Actualiza el estado del documento de titulación.
     * Estados permitidos: 0 pendiente, 1 aprobado, 2 rechazado, 3 observado, 4 inactivo.
     */
    public function updateestado(Request $request, $idstudentdocument)
    {
        $doc = DocumentoTitulacion::find($idstudentdocument);

        if (!$doc) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Documento no encontrado'
            ], 404);
        }



        if (!$request->has('estado')) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Debe enviar el parámetro estado (0, 1, 2 o 3).'
            ]);
        }

        $estado = $request->input('estado');

        if (!in_array($estado, [0, 1, 2, 3, '0', '1', '2', '3'], true)) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Valor de estado inválido. Solo se acepta 0, 1, 2 o 3.'
            ]);
        }




        try {
            $doc->estado = (int) $estado;

            if ($doc->save()) {
                return response()->json([
                    "status"  => true,
                    "mensaje" => 'Estado actualizado correctamente.',
                    "estado"  => $doc->estado
                ]);
            }

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error: el estado no pudo ser actualizado.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al actualizar el estado.',
                "error"   => $e->getMessage()
            ]);
        }
    }

    /**
     * Actualiza el permiso de observación del documento.
     * Solo se puede modificar si el estado del documento es OBSERVADO (3).
     * Solo permite valores 0 o 1.
     */
    public function updatepermiso_observacion(Request $request, $idstudentdocument)
    {
        $doc = DocumentoTitulacion::find($idstudentdocument);

        // Validar existencia
        if (!$doc) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Documento no encontrado'
            ], 404);
        }

        // Validar estado: solo permitir si estado = 3
        if ((int)$doc->estado !== 3) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Solo se puede modificar el permiso de observación cuando el documento está en estado OBSERVADO (3).',
                "estado_actual" => $doc->estado
            ], 400);
        }

        // Validar parámetro
        if (!$request->has('permiso_observacion')) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Debe enviar el parámetro permiso_observacion (0 o 1).'
            ]);
        }

        $permiso = $request->input('permiso_observacion');

        // Validar valores permitidos
        if (!in_array($permiso, [0, 1, '0', '1'], true)) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Valor de permiso_observacion inválido. Solo se acepta 0 o 1.'
            ]);
        }

        // Guardar
        try {
            $doc->permiso_observacion = (int) $permiso;

            if ($doc->save()) {
                return response()->json([
                    "status"              => true,
                    "mensaje"             => 'Permiso de observación actualizado correctamente.',
                    "permiso_observacion" => $doc->permiso_observacion
                ]);
            }

            return response()->json([
                "status"  => false,
                "mensaje" => 'Error: el permiso de observación no pudo ser actualizado.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                "status"  => false,
                "mensaje" => 'Error inesperado al actualizar el permiso de observación.',
                "error"   => $e->getMessage()
            ]);
        }
    }


    /**
     * Ver documento de titulación (PDF o imagen) en línea.
     * Usa la misma lógica que verDocumentoPlan/funcion_verArchivo.
     */
    public function verDocumentoTitulacion($idstudentdocument)
    {
        $doc = DocumentoTitulacion::find($idstudentdocument);

        if (!$doc || !$doc->archivo) {
            // Sin archivo: devolver imagen por defecto
            $file = storage_path('app/public/sinImagen.jpg');
            return $this->verArchivo($file);
        }

        $file = storage_path('app/' . $doc->archivo);

        if (file_exists($file)) {
            return $this->verArchivo($file);
        } else {
            $file = storage_path('app/public/sinImagen.jpg');
            return $this->verArchivo($file);
        }
    }

    /**
     * Función reutilizable para mostrar PDF/JPG/PNG en el navegador (inline).
     */
    public function verArchivo($file)
    {
        $extension = pathinfo($file, PATHINFO_EXTENSION);

        switch (strtolower($extension)) {
            case 'pdf':
                $contentType = 'application/pdf';
                break;
            case 'png':
                $contentType = 'image/png';
                break;
            case 'jpg':
                $contentType = 'image/jpg';
                break;
            case 'jpeg':
                $contentType = 'image/jpeg';
                break;
            default:
                abort(404, 'Formato de archivo no admitido.');
        }

        $headers = [
            'Content-Type' => $contentType,
        ];

        $fileTitle = 'documento.' . $extension;
        return response()->download($file, $fileTitle, $headers, 'inline');
    }
}
