<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;

use App\Http\Controllers\sidebar\SidebarController;
use App\Models\Semesterenrollment;
use App\Models\Pago;
use App\Models\Concepto;
use App\Models\Semester;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use \Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;

class SemesterenrollmentController extends Controller
{
    //
    public function index(Request $request)
    {
        return view('admin.matricula.alumnos_semestre.index');
    }

    public function getInformacion($id)
    {
        // Obtener la información del semestre por su ID
        $semestre = Semester::find($id);

        if ($semestre) {
            // Retornar la información como un JSON con mensaje de éxito
            return response()->json([
                'status' => 'success', // Indicador de éxito
                'message' => 'Información cargada con éxito',
                'informacion' => $semestre->informacion
            ]);
        }

        // En caso de que no se encuentre el semestre
        return response()->json([
            'status' => 'error',
            'message' => 'Semestre no encontrado'
        ], 404);
    }

    public function list(Request $request)
    {
        $semester = Semester::where('estado', 1)->first();
        $idplan = $request->input('idplan');

        if ($request->ajax()) {
            // query
            $data = DB::table('estudiantes as st')
                ->crossJoin('semestres as se')
                ->leftJoin('matriculas_semestres as seen', function ($join) {
                    $join->on('st.id', '=', 'seen.student_id')
                        ->on('se.id', '=', 'seen.semester_id');
                })
                ->join('usuarios as us', 'st.user_id', '=', 'us.id')
                ->join('tipos_identificaciones as it', 'it.id', '=', 'us.identificationtype_id')
                ->join('planes as pl', 'st.plan_id', '=', 'pl.id')
                ->join('programas as pr', 'pl.program_id', '=', 'pr.id')
                ->leftJoin('pagos as pa', function ($join) {
                    $join->on('pa.student_id', '=', 'st.id')
                        ->on('pa.semester_id', '=', 'se.id');
                })
                ->leftJoin('documentos_estudiantes as does', function ($join) {
                    $join->on('does.student_id', '=', 'st.id')
                        ->where('does.estado', 1);
                })
                ->select(
                    DB::raw("CONCAT(se.anho, '-', se.numero) AS semestre"), 
                    'se.id as idsemester',
                    'seen.id as idsemesterenrollment',
                    'seen.student_id',
                    'seen.fecha as fechamatricula',
                    DB::raw("DATE_FORMAT(seen.fecha, '%d/%m/%Y') as fechamatricula2"),
                    'pa.estado as estadopago',
                    'st.id as idstudent',
                    'st.plan_id as idplan',
                    DB::raw('CONCAT(us.nroidenti) as identificacion'),
                    DB::raw('CONCAT(us.apellido_pa, " ", us.apellido_ma, ", ", us.nombres) as estudiante'),
                    DB::raw("CASE WHEN seen.student_id IS NOT NULL THEN '1' ELSE '0' END as relacion_estado"),
                    DB::raw("COALESCE(does.id, 0) as estado_licencia"),
                    DB::raw("COALESCE(does.fecha_inicio, NULL) as fecha_inicio"),
                    DB::raw("CASE WHEN DATE_ADD(does.fecha_inicio, INTERVAL 2 YEAR) >= CURDATE() THEN 'VIGENTE' ELSE 'VENCIDO' END as estado_vigencia")
                )
                ->where('se.id', $semester->id)
                // ->where('st.plan_id', $idplan)
                ->where('st.plan_id', 2)
                ->where('st.estado', 1)
                ->orderBy('us.apellido_pa')
                ->orderBy('us.apellido_ma')
                ->orderBy('us.nombres')
                ->get();

            // datatable
            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }
        return  response()->json(["status" => false, "mensaje" => 'Error: no se pueden cargar los archivos']);
    }

    public function list_courses_by_student(Request $request)
    {
        $semester = Semester::where('estado', 1)->first();
        $idstudent = $request->input('idstudent');

        if ($request->ajax()) {
            // query
            $data = DB::table('asignaturas as su')
                ->join('matriculas_semestres as seen', 'su.semester_id', '=', 'seen.semester_id')
                ->join('matriculas_asignaturas as suen', function ($join) {
                    $join->on('su.id', '=', 'suen.subject_id')
                        ->on('seen.student_id', '=', 'suen.student_id');
                })
                ->join('estudiantes as st', 'seen.student_id', '=', 'st.id')
                ->join('cursos as co', 'su.course_id', '=', 'co.id')
                ->join('periodos as pe', 'co.period_id', '=', 'pe.id')
                ->leftJoin('docentes as te', 'su.teacher_id', '=', 'te.id')
                ->leftJoin('usuarios as us2', 'te.user_id', '=', 'us2.id')
                ->select(
                    'st.id as idstudent',
                    'co.id as idcourse',
                    'co.tipo as tipocourse',
                    'co.codcurso',
                    'co.nombre as nombrecourse',
                    'co.codpredecesor',
                    'co.horas',
                    'co.creditos',
                    'pe.numero as periodo',
                    'su.id as idsubject',
                    'su.seccion',
                    'su.turno',
                    'su.tipo as tiposubject',
                    'su.nota_minima',
                    'te.id as idteacher',
                    DB::raw('CONCAT(us2.apellido_pa, " ", us2.apellido_ma, ", ", us2.nombres) AS docente'),

                )
                ->where('su.semester_id', $semester->id)
                ->where('seen.student_id', $idstudent)
                ->orderBy('pe.numero', 'ASC')
                ->orderBy('co.nombre', 'ASC')
                ->get();

                // 2. Para cada curso, agrega el promedio final usando tu función
                $data->transform(function($curso) use ($idstudent, $semester) {
                    $promedio = $this->get_promedio_final($curso->idsubject, $idstudent, $semester->id);
                    $curso->promedio_final = $promedio;
                    return $curso;
                });

            // datatable
            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }
        return  response()->json(["status" => false, "mensaje" => 'No se pueden cargar los datos']);
    }


    public function list_courses_by_student_for_enrollment(Request $request)
    {
        $semester = Semester::where('estado', 1)->first();
        $idstudent = $request->input('idstudent');

        if ($request->ajax()) {
            // query
            $data = DB::table('asignaturas as su')
                ->join('matriculas_semestres as seen', 'su.semester_id', '=', 'seen.semester_id')
                ->join('matriculas_asignaturas as suen', function ($join) {
                    $join->on('su.id', '=', 'suen.subject_id')
                        ->on('seen.student_id', '=', 'suen.student_id');
                })
                ->join('estudiantes as st', 'seen.student_id', '=', 'st.id')
                ->join('cursos as co', 'su.course_id', '=', 'co.id')
                ->join('periodos as pe', 'co.period_id', '=', 'pe.id')
                ->leftJoin('docentes as te', 'su.teacher_id', '=', 'te.id')
                ->leftJoin('usuarios as us2', 'te.user_id', '=', 'us2.id')
                ->select(
                    'st.id as idstudent',
                    'co.id as idcourse',
                    'co.tipo as tipocourse',
                    'co.codcurso',
                    'co.nombre as nombrecourse',
                    'co.codpredecesor',
                    'co.horas',
                    'co.creditos',
                    'pe.numero as periodo',
                    'su.id as idsubject',
                    'su.seccion',
                    'su.turno',
                    'su.tipo as tiposubject',
                    'su.nota_minima',
                    'te.id as idteacher',
                    DB::raw('CONCAT(us2.apellido_pa, " ", us2.apellido_ma, ", ", us2.nombres) AS docente'),

                )
                ->where('su.semester_id', $semester->id)
                ->where('seen.student_id', $idstudent)
                ->orderBy('pe.numero', 'ASC')
                ->orderBy('co.nombre', 'ASC')
                ->get();

                // 2. Para cada curso, agrega el promedio final usando tu función
                $data->transform(function($curso) use ($idstudent, $semester) {
                    $promedio = $this->get_promedio_final($curso->idsubject, $idstudent, $semester->id);
                    $curso->promedio_final = $promedio;
                    return $curso;
                });

            // datatable
            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }
        return  response()->json(["status" => false, "mensaje" => 'No se pueden cargar los datos']);
    }


    public function get_promedio_final($idsubject, $idstudent, $idsemester)
    {
        // 1. Promedio por actividad
        $promediosActividades = DB::table('matriculas_asignaturas AS ma')
            ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
            ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
            ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
            ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
            ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
            ->leftJoin('notas_estudiantes AS ne', function ($join) {
                $join->on('ne.grade_id', '=', 'n.id')
                    ->on('ne.subjectenrollment_id', '=', 'ma.id');
            })
            ->where('ma.student_id', $idstudent)
            ->where('a.id', $idsubject)
            ->where('se.id', $idsemester)
            ->groupBy('i.id', 'ac.id')
            ->select([
                'i.id as idindicator',
                'ac.id as idactivity',
                DB::raw('COALESCE(ROUND(SUM(ne.nota * n.porcentaje), 1), 0) as promedio_actividad')
            ])
            ->get();

        // 2. Promedio por indicador
        $promediosIndicadores = [];
        if (!$promediosActividades->isEmpty()) {
            $grupos = $promediosActividades->groupBy('idindicator');
            foreach ($grupos as $idindicator => $actividades) {
                $suma = $actividades->sum('promedio_actividad');
                $count = $actividades->count();
                $promediosIndicadores[] = $count > 0 ? round($suma / $count, 1) : 0;
            }
        }

        // 3. Promedio final (promedio de promedios de indicadores)
        if (count($promediosIndicadores) > 0) {
            $promedioFinal = round(array_sum($promediosIndicadores) / count($promediosIndicadores), 1);
            return $promedioFinal;
        }

        return 0;
    }

    public function store(Request $request)
    {
        $semester = Semester::where('estado', 1)
            ->whereDate('fecha_inicio_matricula', '<=', Carbon::now())
            ->whereDate('fecha_fin_matricula', '>=', Carbon::now())
            ->first();

        if (!$semester) {
            // No hay semestre activo con matrícula vigente hoy
            return response()->json([
                "status"  => false,
                "mensaje" => 'Las matrículas no están habilitadas'
            ]);
        }

        $idstudent = $request->input('txt-idstudent');

        $item = new Semesterenrollment();
        $item->semester_id = $semester->id;
        $item->student_id = $idstudent;
        $item->docboucher = "--";
        $item->estado = 0;
        $item->fecha = $request->input('txt-fecha');

        $item_concepto = Concepto::find(1);

        $item_pago = new Pago();
        $item_pago->semester_id = $semester->id;
        $item_pago->student_id = $request->input('txt-idstudent');
        $item_pago->concepto_id = 1;
        $item_pago->fecha = $request->input('txt-fecha');
        $item_pago->monto_inicial = $item_concepto->monto;
        $item_pago->administrator_id = Auth::user()->administradores[0]->id;
        $item_pago->descuento = 0;
        $item_pago->monto_final = $item_concepto->monto;
        $item_pago->documento = "--";
        $item_pago->estado = 0;

        if ($item->save() && $item_pago->save() ) {
            return response()->json(["status" => true, "mensaje" => 'Matricula registrada correctamente']);
        } else {
            return response()->json(["status" => false, "mensaje" => 'Matricula no se pudo registrar']);
        }
    }


    public function update(Request $request, $idsemesterenrollment)
    {
        $semester = Semester::where('estado', 1)
            ->whereDate('fecha_inicio_matricula', '<=', Carbon::now())
            ->whereDate('fecha_fin_matricula', '>=', Carbon::now())
            ->first();

        if (!$semester) {
            // No hay semestre activo con matrícula vigente hoy
            return response()->json([
                "status"  => false,
                "mensaje" => 'Las matrículas no están habilitadas'
            ]);
        }

        $item = Semesterenrollment::find($idsemesterenrollment);

        if (!$item) {
            abort(404);
        } else {
            // recoger datos
            $item->fecha = $request->input('txt-fecha');

            if ($item->save()) {
                return response()->json(["status" => true, "mensaje" => 'Registro actualizado']);
            } else {
                return response()->json(["status" => false, "mensaje" => 'Registro no actualizado']);
            }
        }
    }


    public function destroy($idsemesterenrollment)
    {
        // Busca el registro por su ID
        $item = Semesterenrollment::find($idsemesterenrollment);

        // recuperar data
        $idstudent = $item->student_id;
        $idsemester = $item->semester_id;

        $item_pago = DB::table('pagos')->where('semester_id', $idsemester)->where('student_id', $idstudent);
        $ruta = $item_pago->first()->documento;
        $file = storage_path('app/' . $ruta);
        if (File::exists($file)) {
            File::delete($file);
        }

        // Verifica si el registro existe
        if (!$item) {
            return response()->json(["status" => false, "mensaje" => 'El registro no existe']);
        }

        // Elimina el registro de la base de datos
        if ($item->delete() && $item_pago->delete()) {
            return response()->json(["status" => true, "mensaje" => 'El registro fue eliminado']);
        } else {
            return response()->json(["status" => false, "mensaje" => 'El registro no fue eliminado']);
        }
    }
}
