<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;

use App\Http\Controllers\sidebar\SidebarController;
use App\Models\Exoneration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use \Illuminate\Support\Facades\Auth;
use App\Models\Pago;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ExonerationController extends Controller
{

    public function index(Request $request)
    {
        return view('admin.registros_pagos.exoneraciones.index');
    }


    public function list(Request $request)
    {
        if ($request->ajax()) {
            $data = Exoneration::select([
                'id',
                'nombre',
                'descripcion',
                'documento_resolucion',
                'porcentaje_descuento',
            ])
                ->get();

            return \Yajra\DataTables\DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return  response()->json([
            "status" => false,
            "mensaje" => 'Error: no se pueden cargar los archivos'
        ]);
    }


    public function store(Request $request)
    {
        // 1. Definir reglas de validación que coincidan con los 'name' del formulario
        $rules = [
            'txt-nombre' => 'required|string|max:150',
            'txt-descripcion' => 'nullable|string',
            'documento_resolucion' => 'nullable|file|mimes:pdf|max:2048',
            'txt-porcentaje_descuento' => 'required|numeric|between:0,100',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(["status" => false, "mensaje" => $validator->errors()->first()]);
        }

        try {
            // 2. Crear el registro SIN el path del documento
            $item = new Exoneration();
            $item->nombre = $request->input('txt-nombre');
            $item->descripcion = $request->input('txt-descripcion');
            $item->porcentaje_descuento = $request->input('txt-porcentaje_descuento');
            $item->save(); // Guardar para obtener el ID

            // 3. Si hay archivo, nombrarlo con el ID y guardarlo
            if ($request->hasFile('documento_resolucion')) {
                $file = $request->file('documento_resolucion');
                $extension = $file->getClientOriginalExtension();
                $fileName = 'resolucion_' . $item->id . '.' . $extension;

                $path = $file->storeAs('resoluciones', $fileName, 'public');

                // 4. Actualizar el registro con la ruta del archivo
                $item->documento_resolucion = $path;
                $item->save();
            }

            return response()->json(["status" => true, "mensaje" => 'Registro insertado con éxito']);
        } catch (\Exception $e) {
            return response()->json(["status" => false, "mensaje" => 'Error al guardar el registro.']);
        }
    }

    public function ajax_all(Request $request)
    {
        if ($request->ajax()) {
            $data = Exoneration::all();
            return response()->json($data);
        }
        return  response()->json([
            "status" => false,
            "mensaje" => 'Error en request'
        ]);
    }


    public function update(Request $request, $id)
    {
        $item = Exoneration::find($id);
        if (!$item) {
            return response()->json(["status" => false, "mensaje" => 'Registro no encontrado']);
        }

        // 1. Definir reglas de validación para la actualización
        $rules = [
            'txt-nombre' => 'required|string|max:150',
            'txt-descripcion' => 'nullable|string',
            'documento_resolucion' => 'nullable|file|mimes:pdf|max:2048',
            'txt-porcentaje_descuento' => 'required|numeric|between:0,100',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return response()->json(["status" => false, "mensaje" => $validator->errors()->first()]);
        }

        try {
            // 2. Mapear los datos del request a los campos del modelo
            $item->nombre = $request->input('txt-nombre');
            $item->descripcion = $request->input('txt-descripcion');
            $item->porcentaje_descuento = $request->input('txt-porcentaje_descuento');

            if ($request->hasFile('documento_resolucion')) {
                if ($item->documento_resolucion) {
                    Storage::disk('public')->delete($item->documento_resolucion);
                }
                $file = $request->file('documento_resolucion');
                $extension = $file->getClientOriginalExtension();
                $fileName = 'resolucion_' . $item->id . '.' . $extension;
                $path = $file->storeAs('resoluciones', $fileName, 'public');
                $item->documento_resolucion = $path;
            }

            $item->save();

            return response()->json(["status" => true, "mensaje" => 'Registro actualizado con éxito']);
        } catch (\Exception $e) {
            return response()->json(["status" => false, "mensaje" => 'Error al actualizar el registro.']);
        }
    }


    public function destroy($id)
    {
        $item = Exoneration::find($id);
        if (!$item) {
            return response()->json(["status" => false, "mensaje" => "El registro no existe"]);
        }

        $hasPayments = Pago::where('exoneration_id', $item->id)->exists();
        if ($hasPayments) {
            return response()->json(["status" => false, "mensaje" => "No se puede eliminar porque tiene pagos asociados"]);
        }

        if ($item->documento_resolucion) {
            Storage::disk('public')->delete($item->documento_resolucion);
        }

        $deleted = $item->delete();
        return response()->json([
            "status" => (bool) $deleted,
            "mensaje" => $deleted ? "El registro fue eliminado" : "El registro no fue eliminado"
        ]);
    }

    public function viewDocument($id)
    {
        $exoneration = Exoneration::find($id);

        // 1. Validar que el registro y el archivo existan
        if ($exoneration && !empty($exoneration->documento_resolucion) && Storage::disk('public')->exists($exoneration->documento_resolucion)) {

            // 2. Obtener la ruta completa del archivo en el servidor
            $path = Storage::disk('public')->path($exoneration->documento_resolucion);

            // 3. Servir el archivo para que el navegador lo muestre
            return response()->file($path);

        } else {
            // Si no se encuentra, muestra un error
            return response('<html><body><script>alert("Error: El documento no fue encontrado o no ha sido cargado."); window.close();</script></body></html>', 404);
        }
    }
}
