<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ConvalidationsController extends Controller
{
    public function index(Request $request)
    {
        return view('admin.estudiantes.convalidaciones.index');
    }

    public function ajaxPrograms()
    {
        try {
            $programs = DB::table('programs')
                ->whereNull('deleted_at')
                ->select('id', 'codprograma', 'nombre')
                ->orderBy('nombre')
                ->get();

            return response()->json($programs);

        } catch (\Throwable $e) {
            Log::error('Error en ajaxPrograms', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => false,
                'message' => 'Error al cargar los programas.',
            ], 500);
        }
    }

    public function studentByProgram($idprogram)
    {
        $rows = DB::table('usuarios as u')
            ->join('estudiantes as e', 'e.user_id', '=', 'u.id')
            ->join('planes as p', 'p.id', '=', 'e.plan_id')
            ->join('traslados as t', 't.student_id', '=', 'e.id')
            ->where('p.program_id', $idprogram)
            ->selectRaw('
                e.id as idstudent,
                u.apellido_pa, u.apellido_ma, u.nombres,
                u.identificationtype_id as tipoidenti,
                u.nroidenti,
                u.celular,
                e.anho_ingreso,
                e.estado,
                p.id as plan_id
            ')
            ->orderBy('u.apellido_pa')
            ->orderBy('u.apellido_ma')
            ->orderBy('u.nombres')
            ->get();

        return response()->json($rows);
    }

    public function listar(Request $request)
    {
        $idStudent = $request->query('idstudent');
        if (! $idStudent || $idStudent === '0') {
            return response()->json([]);
        }

        try {
            $rows = DB::table('convalidaciones as ce')
                ->join('traslados as t', 'ce.transfers_id', '=', 't.id')
                ->leftJoin('instituciones as io', 't.institucion_origen', '=', 'io.id')
                ->leftJoin('instituciones as id', 't.institucion_destino', '=', 'id.id')

                ->where('t.student_id', $idStudent)
                ->whereNull('ce.deleted_at')
                ->select(
                    'ce.id',
                    'ce.documento',
                    'ce.fecha_registro',
                    'ce.created_at',
                    't.student_id',

                    DB::raw("CASE WHEN t.tipo = 0 THEN 'Ingreso' WHEN t.tipo = 1 THEN 'Salida' ELSE 'N/A' END as tipo_nombre"),

                    DB::raw("CASE WHEN t.tipo = 0 THEN io.name WHEN t.tipo = 1 THEN id.name ELSE 'N/A' END as institucion_name")
                )
                ->orderByDesc('ce.created_at')
                ->get();

            foreach ($rows as $r) {
                $r->documento_url = $r->documento
                    ? route('documentos', ['file' => $r->documento])
                    : null;
            }

            return response()->json($rows, 200, [], JSON_UNESCAPED_UNICODE);

        } catch (\Throwable $e) {
            Log::error('listar convalidaciones', [
                'msg' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status' => false,
                'message' => 'Error al listar convalidaciones externas.',
                'detail' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'fecha_registro' => 'required|date',
                'documento' => 'nullable|file|mimes:pdf',
            ]);

            // Verificar que el estudiante existe
            $estudiante = DB::table('estudiantes as e')
                ->join('usuarios as u', 'u.id', '=', 'e.user_id')
                ->where('e.id', $request->student_id)
                ->select('u.apellido_pa', 'u.apellido_ma', 'u.nombres')
                ->first();

            if (! $estudiante) {
                throw new \Exception('Estudiante no encontrado.');
            }

            // Verificar que la institución existe
            $traslado = DB::table('traslados')
                ->where('student_id', $request->student_id)
                ->orderByDesc('fecha')
                ->first();

            if (! $traslado) {
                throw new \Exception('Traslado no encontrado.');
            }

            $transferId = $traslado->id;

            $nombreEstudiante = Str::slug(
                trim("{$estudiante->apellido_pa}_{$estudiante->apellido_ma}_{$estudiante->nombres}")
            );

            $documentoPath = null;

            if ($request->hasFile('documento')) {
                $file = $request->file('documento');
                $studentId = $request->student_id;
                $directoryPath = "students/{$studentId}/convalidaciones";
                $extension = $file->getClientOriginalExtension();
                $fileName = "Convalidacion-{$nombreEstudiante}".".{$extension}";

                $documentoPath = $file->storeAs($directoryPath, $fileName, 'public');
            }
            // Insertar en la base de datos
            DB::table('convalidaciones')->insert([
                'transfers_id' => $transferId,
                'fecha_registro' => $validated['fecha_registro'],
                'documento' => $documentoPath,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            DB::commit();

            return response()->json([
                'status' => true,
                'mensaje' => 'Convalidación externa registrada exitosamente',
                'documento_url' => $documentoPath ? route('documentos', ['file' => $documentoPath]) : null,
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return response()->json([
                'status' => false,
                'mensaje' => 'Error de validación',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Error storing convalidacion externa', [
                'error' => $e->getMessage(),
                'student_id' => $request->student_id,
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status' => false,
                'mensaje' => 'Error al guardar la convalidación externa: '.$e->getMessage(),
            ], 500);
        }
    }

    // ver documento (pdf, doc, docx, jpg, png)
    public function verDocumento($file)
    {
        $file = str_replace(['../', '..\\'], '', $file); // Previene path traversal
        $file = ltrim($file, '/'); // Remover slashes iniciales

        $path = storage_path('app/public/'.$file);

        if (! file_exists($path)) {
            abort(404, 'Documento no encontrado.');
        }

        $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        $mimeMap = [
            'pdf' => 'application/pdf',
            'doc' => 'application/msword',
            'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
        ];

        $mime = $mimeMap[$ext] ?? 'application/octet-stream';

        if (in_array($ext, ['pdf', 'jpg', 'jpeg', 'png'])) {
            return response()->file($path, ['Content-Type' => $mime]);
        }

        // Para otros tipos (DOC, DOCX): forzar descarga
        return response()->download($path, basename($file), ['Content-Type' => $mime]);
    }

    //formulario de registro de cursos de convalidación externa
    public function storeConvalidacionCurso(Request $request)
    {
        $validated = $request->validate([
            'convalidation_id' => 'required|exists:convalidaciones,id',
            'course_id' => 'required|exists:cursos,id',
            'code' => 'required|string|max:50',
            'name' => 'required|string|max:255',
            'credits' => 'required|integer|min:1|max:99',
            'final_average' => 'nullable|numeric|min:0|max:20',
            'observation' => 'nullable|string|max:500',
        ]);

        DB::table('convalidacion_cursos')->insert([
            'convalidation_id' => $validated['convalidation_id'],
            'course_id' => $validated['course_id'],
            'code' => strtoupper($validated['code']),
            'name' => $validated['name'],
            'credits' => $validated['credits'],
            'final_average' => $validated['final_average'] ?? null,
            'observation' => $validated['observation'] ?? null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return response()->json([
            'status' => true,
            'mensaje' => 'Curso externo registrado correctamente',
        ]);
    }

    public function listCourseEquivalences($convalidationId)
    {
        try {
            Log::info('ListCourseEquivalences called with ID: '.$convalidationId);

            if ($convalidationId == '0') {
                return response()->json([]);
            }

            // Verificar si la convalidación existe
            $convalidationExists = DB::table('convalidaciones')
                ->where('id', $convalidationId)
                ->exists();

            if (! $convalidationExists) {
                return response()->json([
                    'status' => false,
                    'mensaje' => 'Convalidación no encontrada',
                ], 404);
            }

            $equivalences = DB::table('convalidacion_cursos as cc')
                ->join('cursos as c', 'cc.course_id', '=', 'c.id')
                ->where('cc.convalidation_id', $convalidationId)
                ->whereNull('cc.deleted_at')
                ->select(
                    'cc.id',
                    'cc.codigo',
                    'c.nombre as course_name',
                    'cc.creditos',
                    'cc.horas_por_semana',
                    'cc.promedio_final',
                    'cc.observaciones',
                    'cc.created_at'
                )
                ->orderBy('cc.id')
                ->get();

            Log::info('Equivalences found: '.$equivalences->count());

            return response()->json($equivalences);
        } catch (\Throwable $e) {
            Log::error('Error in listCourseEquivalences', [
                'error' => $e->getMessage(),
                'convalidation_id' => $convalidationId,
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'status' => false,
                'mensaje' => 'Error al cargar las equivalencias',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function storeCourseEquivalence(Request $request)
    {
        try {
            DB::beginTransaction();

            $validated = $request->validate([
                'convalidation_id' => 'required|exists:convalidaciones,id',
                'course_id' => 'required|exists:cursos,id',
                'code' => 'required|string|max:50',
                'name' => 'required|string|max:255',
                'credits' => 'required|integer|min:1|max:99',
                'hours_per_week' => 'required|integer|min:1|max:40',
                'final_average' => 'nullable|numeric|min:0|max:20',
                'observation' => 'nullable|string|max:500',
            ]);

            DB::table('convalidacion_cursos')->insert([
                'convalidation_id' => $validated['convalidation_id'],
                'course_id' => $validated['course_id'],
                'codigo' => $validated['code'],
                'nombre' => $validated['name'],
                'creditos' => $validated['credits'],
                'horas_por_semana' => $validated['hours_per_week'],
                'promedio_final' => $validated['final_average'] ?? null,
                'observaciones' => $validated['observation'] ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::commit();

            return response()->json([
                'status' => true,
                'mensaje' => 'Equivalencia de curso registrada exitosamente',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return response()->json([
                'status' => false,
                'mensaje' => 'Error de validación',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('Error storing course equivalence', ['error' => $e->getMessage()]);

            return response()->json([
                'status' => false,
                'mensaje' => 'Error al guardar la equivalencia: '.$e->getMessage(),
            ], 500);
        }
    }

    public function ajaxCursosByPlan($idplan)
    {
        $cursos = DB::table('cursos')
            ->join('planes', 'cursos.period_id', '=', 'planes.id')
            ->where('planes.id', $idplan)
            ->select('cursos.id', 'cursos.codcurso as codigo_curso', 'cursos.nombre')
            ->orderBy('cursos.nombre')
            ->get();

        return response()->json($cursos);
    }

    public function destroyCourseEquivalence($id)
    {
        $updated = DB::table('convalidacion_cursos')
            ->where('id', $id)
            ->update(['deleted_at' => now(), 'updated_at' => now()]);

        if ($updated) {
            return response()->json(['status' => true, 'mensaje' => 'Equivalencia eliminada correctamente']);
        }

        return response()->json(['status' => false, 'mensaje' => 'No se encontró el registro'], 404);
    }

    public function ajaxInstitutions()
    {
        $inst = DB::table('instituciones')
            ->select('id', 'name')
            ->get();

        return response()->json($inst);
    }

    public function destroyConvalidacion($id)
    {
        $updated = DB::table('convalidaciones')
            ->where('id', $id)
            ->update(['deleted_at' => now(), 'updated_at' => now()]);

        if ($updated) {
            return response()->json(['status' => true, 'mensaje' => 'Convalidación eliminada correctamente']);
        }

        return response()->json(['status' => false, 'mensaje' => 'No se encontró la convalidación'], 404);
    }
}
