<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;

use App\Http\Controllers\sidebar\SidebarController;
use App\Models\Administrator;
use App\Models\Student;
use App\Models\Subject;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf as PDF;
use Illuminate\Support\Facades\Validator;
use \Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Carbon;

use App\Providers\AppServiceProvider;

class ConsultationController extends Controller
{
    //
    public function index(Request $request)
    {
        return view('admin.consulta.constancias_certificados.index');
    }

    public function index2(Request $request)
    {
        return view('admin.consulta.actas_oficiales.index');
    }

    public function index3(Request $request)
    {
        return view('admin.consulta.indicadores_gestion.index');
    }


    public function index4(Request $request)
    {
        return view('admin.consulta.primeros_puestos.index');
    }


    public function list_estudiantes(Request $request, $idplan)
    {
        if ($request->ajax()) {
            // query
            $data = DB::table('estudiantes')
                ->join('usuarios', 'usuarios.id', '=', 'estudiantes.user_id')
                ->join('tipos_identificaciones AS it', 'it.id', '=', 'usuarios.identificationtype_id')
                ->join('planes', 'planes.id', '=', 'estudiantes.plan_id')
                ->join('programas', 'programas.id', '=', 'planes.program_id')
                ->select(
                    DB::raw('ROW_NUMBER() OVER (ORDER BY usuarios.apellido_pa) AS n'),
                    'it.id AS ididentificationtype',
                    'it.tipo AS tipoidenti',
                    'usuarios.nroidenti',
                    DB::raw("CONCAT(usuarios.apellido_pa, ' ', usuarios.apellido_ma, ', ', usuarios.nombres) AS estudiante"),
                    'estudiantes.id as idstudent',
                    'estudiantes.anho_ingreso',
                    'estudiantes.estado',
                    DB::raw("CONCAT('Plan ', planes.nombre, ' (', planes.tipo, ')') AS plan"),
                    'programas.nombre AS programa'
                )
                ->where('planes.id', '=', $idplan) // Aca iria el id del plan
                // ->whereNotIn('estudiantes.estado', [5, 6])
                ->orderBy('estudiante')
                ->get();

            return response()->json($data);
        }

        return  response()->json(["status" => false, "mensaje" => 'No se puede cargar la información']);
    }

    public function list_record_academico(Request $request)
    {
        $idStudent = $request->input('idStudent');

        if ($request->ajax()) {

            // Subconsulta gradesQuery
            $gradesQuery = DB::table('estudiantes AS es')
                ->join('matriculas_semestres AS ms', 'ms.student_id', '=', 'es.id')
                ->join('matriculas_asignaturas AS ma', 'ma.student_id', '=', 'ms.student_id')
                ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
                ->join('cursos AS c', 'c.id', '=', 'a.course_id')
                ->join('periodos AS pe', 'pe.id', '=', 'c.period_id')
                ->join('planes AS pl', 'pl.id', '=', 'pe.plan_id')
                ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
                ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
                ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
                ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
                ->leftJoin('notas_estudiantes AS ne', function ($join) {
                    $join->on('ne.grade_id', '=', 'n.id')
                        ->on('ne.subjectenrollment_id', '=', 'ma.id');
                })
                ->selectRaw('
                    es.id AS idstudent,
                    pl.nombre AS plan,
                    c.codcurso AS codcurso,
                    c.nombre AS curso,
                    CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) AS creditos_horas,
                    pe.numero AS periodo,
                    a.tipo AS condicion,
                    a.seccion AS seccion,
                    a.turno AS turno,
                    CONCAT(se.anho, " - ", se.numero) AS semestre,
                    i.id AS idindicator,
                    ac.id AS idactivity,
                    n.id AS idgrade,
                    ne.id AS idstudentgrade,
                    n.nombre AS tipo_nota,
                    n.porcentaje AS porcentaje,
                    COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 1), 0) AS nota
                ')
                ->where('ma.student_id', '=', $idStudent)
                ->groupBy([
                    'plan',
                    'curso',
                    'codcurso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'semestre',
                    'idindicator',
                    'idactivity',
                    'idgrade',
                    'idstudentgrade',
                    'tipo_nota',
                    'porcentaje',
                    'idstudent'
                ])
                ->orderBy('curso');

            // Subconsulta activityQuery
            $activityQuery = DB::table(DB::raw("({$gradesQuery->toSql()}) AS gradesQuery"))
                ->mergeBindings($gradesQuery)
                ->selectRaw('
                    plan,
                    codcurso,
                    curso,
                    creditos_horas,
                    periodo,
                    condicion,
                    seccion,
                    turno,
                    semestre,
                    idindicator,
                    idactivity,
                    COALESCE(ROUND(SUM(nota * porcentaje), 1), 0) AS promedio_actividades
                ')
                ->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre', 'idindicator', 'idactivity']);

            // Subconsulta indicatorQuery
            $indicatorQuery = DB::table(DB::raw("({$activityQuery->toSql()}) AS activityQuery"))
                ->mergeBindings($activityQuery)
                ->selectRaw('
                    plan,
                    codcurso,
                    curso,
                    creditos_horas,
                    periodo,
                    condicion,
                    seccion,
                    turno,
                    semestre,
                    idindicator,
                    COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 1), 0) AS promedio_indicadores
                ')
                ->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre', 'idindicator']);

            // Subconsulta finalGrades
            $finalGrades = DB::table(DB::raw("({$indicatorQuery->toSql()}) AS indicatorQuery"))
                ->mergeBindings($indicatorQuery)
                ->selectRaw('
                    plan,
                    codcurso,
                    curso,
                    creditos_horas,
                    periodo,
                    condicion,
                    seccion,
                    turno,
                    semestre,
                    COALESCE(ROUND(SUM(promedio_indicadores) / COUNT(idindicator), 1), 0) AS promedio_final
                ')
                ->groupBy(['plan', 'curso', 'codcurso', 'creditos_horas', 'periodo', 'condicion', 'seccion', 'turno', 'semestre']);

            // Consulta final
            $result = DB::table('planes AS pl')
                ->join('estudiantes AS es', 'es.plan_id', '=', 'pl.id')
                ->join('periodos AS pe', 'pe.plan_id', '=', 'pl.id')
                ->join('cursos AS c', 'c.period_id', '=', 'pe.id')
                ->join('modulos_formativos AS mf', 'mf.id', '=', 'c.training_module_id')
                ->leftJoinSub($finalGrades, 'fg', function ($join) {
                    $join->on('fg.codcurso', '=', 'c.codcurso')
                        ->on('fg.plan', '=', 'pl.nombre');
                })
                ->selectRaw('
                    es.id AS idstudent,
                    pl.nombre AS plan,
                    mf.id AS idmodulo,
                    mf.numero AS numero_modulo,
                    mf.codigo AS codigo_modulo,
                    mf.descripcion AS descripcion_modulo,
                    pe.numero AS periodo,
                    c.codcurso AS codcurso,
                    c.nombre AS nombre_curso,
                    c.tipo AS tipo_curso,
                    c.creditos AS creditos,
                    c.horas AS horas,
                    fg.seccion AS seccion,
                    fg.turno AS turno,
                    fg.semestre AS semestre,
                    fg.promedio_final AS promedio_final,
                    CASE
                        WHEN fg.promedio_final >= 13 THEN "Aprobado"
                        WHEN fg.promedio_final < 13 AND fg.promedio_final > 0 THEN "Desaprobado"
                        WHEN fg.promedio_final = 0 AND fg.plan IS NOT NULL THEN "Por evaluar"
                        ELSE "--"
                    END AS estado
                ')
                ->orderBy('numero_modulo')
                ->orderBy('periodo')
                ->orderBy('tipo_curso', 'asc')
                ->orderBy('curso')
                ->where('es.id', '=', $idStudent)
                ->get();

            // datatable
            return DataTables::of($result)
                ->addIndexColumn()
                ->make(true);
        }
        return  response()->json(["status" => false, "mensaje" => 'Error: no se pueden cargar los archivos']);
    }

    public function list_periodos(Request $request)
    {
        $idsemester = $request->input('idsemester');

        if ($request->ajax()) {
            // query
            $data = DB::table('programas AS pr')
                ->join('planes AS pl', 'pl.program_id', '=', 'pr.id')
                ->join('periodos AS pe', 'pe.plan_id', '=', 'pl.id')
                ->join('periodos_semestres AS ps', 'ps.period_id', '=', 'pe.id')
                ->join('semestres AS se', 'se.id', '=', 'ps.semester_id')
                ->where('se.id', $idsemester)
                ->select(
                    'pe.id AS idperiod',
                    'se.id AS idsemester',
                    'pr.nombre AS programa',
                    'pl.nombre AS plan',
                    'pe.numero AS periodo'
                )
                ->orderBy('programa')
                ->orderBy('plan', 'desc')
                ->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return  response()->json(["status" => false, "mensaje" => 'No se puede cargar la información']);
    }

    public function acta_matricula($idsemester, $idperiod)
    {
        // Validar si hay semestres vigentes o futuros
        if ($this->ValidacionFecFinSemestres() !== 'OK') {
            return response()->json([
                'status' => false,
                'mensaje' => 'No se puede emitir el certificado: el semestre no ha finalizado.'
            ], 403);
        }

        $info = DB::table('programas AS pr')
            ->join('planes AS pl', 'pl.program_id', '=', 'pr.id')
            ->join('periodos AS pe', 'pe.plan_id', '=', 'pl.id')
            ->join('periodos_semestres AS ps', 'ps.period_id', '=', 'pe.id')
            ->join('semestres AS se', 'se.id', '=', 'ps.semester_id')
            ->where('se.id', $idsemester)
            ->where('pe.id', $idperiod)
            ->select([
                'pr.nombre AS programa',
                'pr.nivel_formativo AS nivel_formativo',
                DB::raw('CONCAT("PLAN ", pl.nombre, " (", pl.tipo, ")") AS plan'),
                'pe.numero AS periodo',
                DB::raw('CONCAT(se.anho, "-", se.numero) AS semestre'),
            ])
            ->first();

        $cursos = DB::table('planes as pl')
            ->join('periodos as pe', 'pe.plan_id', '=', 'pl.id')
            ->join('cursos as cu', 'cu.period_id', '=', 'pe.id')
            ->join('asignaturas as a', 'a.course_id', '=', 'cu.id')
            ->join('semestres as se', 'se.id', '=', 'a.semester_id')
            ->join('periodos_semestres as ps', function ($join) {
                $join->on('ps.period_id', '=', 'pe.id')
                    ->on('ps.semester_id', '=', 'se.id');
            })
            ->where('pe.id', $idperiod)
            ->where('se.id', $idsemester)
            ->orderBy('nombre')
            ->select('cu.nombre as nombre')
            ->get();

        $total_filas = DB::table('periodos as pe')
            ->join('cursos as cu', 'cu.period_id', '=', 'pe.id')
            ->join('asignaturas as a', 'a.course_id', '=', 'cu.id')
            ->join('semestres as se', 'se.id', '=', 'a.semester_id')
            ->join('periodos_semestres as ps', function ($join) {
                $join->on('ps.period_id', '=', 'pe.id')
                    ->on('ps.semester_id', '=', 'se.id');
            })
            ->where('pe.id', $idperiod)
            ->where('se.id', $idsemester)
            ->select(DB::raw('COUNT(cu.nombre) as cantidad'))
            ->first();

        // Aumentar el límite de group_concat
        DB::statement("SET SESSION group_concat_max_len = 1000000");

        // Generar las columnas dinámicas
        $columns = DB::table('periodos AS pe')
            ->join('cursos AS cu', 'cu.period_id', '=', 'pe.id')
            ->join('asignaturas AS a', 'a.course_id', '=', 'cu.id')
            ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
            ->where('pe.id', $idperiod)
            ->where('se.id', $idsemester)
            ->orderBy('cu.nombre')
            ->selectRaw("
                GROUP_CONCAT(
                    CONCAT(
                        'MAX(CASE WHEN asi.id = ',
                        a.id,
                        ' THEN CASE WHEN ma.id IS NOT NULL THEN ''SI'' ELSE ''NO'' END END) AS `',
                        REPLACE(cu.nombre, ' ', '_'),
                        '`'
                    )
                ) AS columns
            ")
            ->value('columns');

        // Validar si las columnas fueron generadas correctamente
        if (!$columns) {
            return response()->json(['error' => 'No se pudieron generar las columnas dinámicas.'], 500);
        }

        // Construir y ejecutar la consulta
        $estudiantes = DB::table('estudiantes AS es')
            ->join('usuarios AS us', 'us.id', '=', 'es.user_id')
            ->join('generos AS ge', 'ge.id', '=', 'us.genre_id')
            ->join('planes AS pl', 'pl.id', '=', 'es.plan_id')
            ->join('periodos AS pe', 'pe.plan_id', '=', 'pl.id')
            ->join('cursos AS cu', 'cu.period_id', '=', 'pe.id')
            ->join('asignaturas AS asi', 'asi.course_id', '=', 'cu.id')
            ->join('semestres AS se', 'se.id', '=', 'asi.semester_id')
            ->leftJoin('matriculas_asignaturas AS ma', function ($join) {
                $join->on('ma.subject_id', '=', 'asi.id')
                    ->on('ma.student_id', '=', 'es.id');
            })
            ->where('pe.id', $idperiod)
            ->where('se.id', $idsemester)
            ->groupBy('us.nroidenti', 'estudiante', 'sexo', 'edad')
            ->orderBy('estudiante')
            ->selectRaw("
                us.nroidenti,
                CONCAT(us.apellido_pa, ' ', us.apellido_ma, ', ', us.nombres) AS estudiante,
                CASE
                    WHEN ge.nombre = 'Masculino' THEN 'M'
                    WHEN ge.nombre = 'Femenino' THEN 'F'
                    ELSE 'O'
                END AS sexo,
                TIMESTAMPDIFF(YEAR, us.fecnac, CURDATE()) AS edad,
                $columns
            ")
            ->get();

        // Generar PDF
        $pdf = PDF::loadView('reportes.prueba', [
            'institutions' => AppServiceProvider::$institutions,
            'estudiantes' => $estudiantes,
            'cursos' => $cursos,
            'info' => $info,
            'total_filas' => $total_filas,
        ]);

        $pdf->setPaper('A4', 'landscape');

        return $pdf->stream('prueba.pdf');
    }

    public function tasa_finalizacion(Request $request)
    {

        if ($request->ajax()) {
            // query
            $data = DB::table('programas as pr')
                ->selectRaw('
                CONCAT(se.anho, " - ", se.numero) AS semestre,
                pr.nombre AS programa,
                COUNT(ms.id) AS cantidad,
                ROUND((COUNT(ms.id) * 100) / COUNT(es.id), 2) AS porcentaje
            ')
                ->leftJoin('planes as pl', 'pl.program_id', '=', 'pr.id')
                ->leftJoin('estudiantes as es', 'es.plan_id', '=', 'pl.id')
                ->crossJoin('semestres as se')
                ->leftJoin('matriculas_semestres as ms', function ($join) {
                    $join->on('ms.student_id', '=', 'es.id')
                        ->on('ms.semester_id', '=', 'se.id');
                })
                ->groupBy('se.anho', 'se.numero', 'pr.nombre')
                ->orderBy('semestre')
                ->orderBy('programa')
                ->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return  response()->json(["status" => false, "mensaje" => 'No se puede cargar la información']);
    }

    public function tasa_abandono(Request $request)
    {
        $idsemester = $request->input('idsemester');

        if ($request->ajax()) {
            // query
            $data = DB::table('programas as pr')
                ->selectRaw('
                CONCAT(se.anho, " - ", se.numero) AS semestre,
                pr.nombre AS programa,
                COUNT(es.id) AS cantidad,
                ROUND((COUNT(es.id) * 100) / (
                    SELECT COUNT(est.id)
                    FROM estudiantes as est
                    LEFT JOIN planes as pln ON pln.id = est.plan_id
                    WHERE pln.program_id = pr.id
                ), 2) AS porcentaje
            ')
                ->leftJoin('planes as pl', 'pl.program_id', '=', 'pr.id')
                ->leftJoin('estudiantes as es', 'es.plan_id', '=', 'pl.id')
                ->crossJoin('semestres as se')
                ->leftJoin('matriculas_semestres as ms', function ($join) {
                    $join->on('ms.student_id', '=', 'es.id')
                        ->on('ms.semester_id', '=', 'se.id');
                })
                ->whereNull('ms.id')
                ->groupBy('se.anho', 'se.numero', 'pr.nombre', 'pr.id')
                ->orderBy('semestre')
                ->orderBy('programa')
                ->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }

        return  response()->json(["status" => false, "mensaje" => 'No se puede cargar la información']);
    }


    public function list_primeros_puestos(Request $request)
    {
        $idsemester = $request->input('idsemester');
        $idprogram = $request->input('idprogram');
        $idplan = $request->input('idplan');

        if ($request->ajax()) {

            $query = DB::table('matriculas_asignaturas AS ma')
                ->select(
                    'a.id AS idsubject',
                    'ma.student_id AS idstudent',
                    'pr.nombre AS programa',
                    'pl.nombre AS plan',
                    DB::raw('CONCAT(us.apellido_pa, " ", us.apellido_ma, ", ", us.nombres) AS estudiante'),
                    'c.codcurso AS codcurso',
                    'c.nombre AS curso',
                    DB::raw('CONCAT("Crd: ", c.creditos, " - Hrs: ", c.horas) AS creditos_horas'),
                    'pe.numero AS periodo',
                    'a.tipo AS condicion',
                    'a.seccion AS seccion',
                    'a.turno AS turno',
                    'i.id AS idindicator',
                    'ac.id AS idactivity',
                    'n.id AS idgrade',
                    'ne.id AS idstudentgrade',
                    'n.nombre AS tipo_nota',
                    'n.porcentaje AS porcentaje',
                    DB::raw('COALESCE(ROUND(SUM(ne.nota) / COUNT(ne.id), 1), 0) AS nota')
                )
                ->join('estudiantes AS es', 'es.id', '=', 'ma.student_id')
                ->join('usuarios AS us', 'us.id', '=', 'es.user_id')
                ->join('asignaturas AS a', 'a.id', '=', 'ma.subject_id')
                ->join('cursos AS c', 'c.id', '=', 'a.course_id')
                ->join('periodos AS pe', 'pe.id', '=', 'c.period_id')
                ->join('planes AS pl', 'pl.id', '=', 'pe.plan_id')
                ->join('programas AS pr', 'pr.id', '=', 'pl.program_id')
                ->join('semestres AS se', 'se.id', '=', 'a.semester_id')
                ->leftJoin('indicadores AS i', 'i.subject_id', '=', 'a.id')
                ->leftJoin('actividades AS ac', 'ac.indicator_id', '=', 'i.id')
                ->leftJoin('notas AS n', 'n.activity_id', '=', 'ac.id')
                ->leftJoin('notas_estudiantes AS ne', function ($join) {
                    $join->on('ne.grade_id', '=', 'n.id')
                        ->on('ne.subjectenrollment_id', '=', 'ma.id');
                })
                ->groupBy(
                    'plan',
                    'curso',
                    'codcurso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'idindicator',
                    'idactivity',
                    'idgrade',
                    'idstudentgrade',
                    'tipo_nota',
                    'porcentaje',
                    'idsubject',
                    'idstudent'
                )
                ->where('se.id', $idsemester);

            // if($idsemester != "all") {
            //     $query->where('se.id', $idsemester);
            // }

            if ($idprogram != "all") {
                $query->where('pr.id', $idprogram);
            }

            if ($idplan != "all") {
                $query->where('pl.id', $idplan);
            }

            // Promedios por actividad
            $activityQuery = DB::query()->fromSub($query, 'grades_query')
                ->select(
                    'idsubject',
                    'idstudent',
                    'programa',
                    'plan',
                    'estudiante',
                    'codcurso',
                    'curso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'idindicator',
                    'idactivity',
                    DB::raw('COALESCE(ROUND(SUM(nota * porcentaje), 1), 0) AS promedio_actividades')
                )
                ->groupBy(
                    'plan',
                    'curso',
                    'codcurso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'idindicator',
                    'idactivity',
                    'idsubject',
                    'idstudent'
                );

            // Promedios por indicador
            $indicatorQuery = DB::query()->fromSub($activityQuery, 'activity_query')
                ->select(
                    'idsubject',
                    'idstudent',
                    'programa',
                    'plan',
                    'estudiante',
                    'codcurso',
                    'curso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'idindicator',
                    DB::raw('COALESCE(ROUND(SUM(promedio_actividades) / COUNT(idactivity), 1), 0) AS promedio_indicadores')
                )
                ->groupBy(
                    'plan',
                    'curso',
                    'codcurso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'idindicator',
                    'idsubject',
                    'idstudent'
                );

            // Promedios finales por asignatura
            $finalGradesQuery = DB::query()->fromSub($indicatorQuery, 'indicator_query')
                ->select(
                    'idsubject',
                    'idstudent',
                    'programa',
                    'plan',
                    'estudiante',
                    'periodo',
                    'codcurso',
                    'curso',
                    'creditos_horas',
                    'condicion',
                    'seccion',
                    'turno',
                    DB::raw('COALESCE(ROUND(SUM(promedio_indicadores) / COUNT(idindicator), 1), 0) AS promedio_final')
                )
                ->groupBy(
                    'plan',
                    'curso',
                    'codcurso',
                    'creditos_horas',
                    'periodo',
                    'condicion',
                    'seccion',
                    'turno',
                    'idsubject',
                    'idstudent'
                );

            // Promedio general por estudiante
            $data = DB::query()->fromSub($finalGradesQuery, 'final_grades')
                ->select(
                    'idstudent',
                    'programa',
                    'plan',
                    'estudiante',
                    DB::raw('COALESCE(ROUND(AVG(promedio_final), 1), 0) AS promedio_general')
                )
                ->groupBy('idstudent', 'programa', 'plan', 'estudiante', 'periodo')
                ->orderBy('promedio_general', 'desc')
                ->orderBy('estudiante')
                ->get();

            return DataTables::of($data)
                ->addIndexColumn()
                ->make(true);
        }
    }
}
