<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Audit;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class AuditController extends Controller
{
    public function index()
    {
        return view('admin.seguridad.auditoria.index');
    }

    public function list(Request $request)
    {
        if ($request->ajax()) {
            $query = Audit::with(['user', 'auditable'])
                ->orderByDesc('created_at'); // siempre del más reciente al más antiguo

            // ==========================
            // Filtros opcionales
            // ==========================

            // Filtro por evento (created / updated / deleted)
            if ($request->filled('event')) {
                $query->where('event', $request->input('event'));
            }

            // Filtro por usuario (ID)
            if ($request->filled('user_id')) {
                $query->where('user_id', $request->input('user_id'));
            }

            // Filtro por nombre de usuario (parcial)
            if ($request->filled('user_name')) {
                $userName = $request->input('user_name');

                $query->whereHas('user', function ($q) use ($userName) {
                    $q->where(function ($sub) use ($userName) {
                        $sub->where('nombres', 'LIKE', "%{$userName}%")
                            ->orWhere('apellido_pa', 'LIKE', "%{$userName}%")
                            ->orWhere('apellido_ma', 'LIKE', "%{$userName}%");
                    });
                });
            }

            // Filtro por tipo de usuario (App\Models\User, etc.)
            if ($request->filled('user_type')) {
                $query->where('user_type', 'LIKE', '%' . $request->input('user_type') . '%');
            }

            // Filtro por modelo / tabla afectada (auditable_type)
            if ($request->filled('auditable_type')) {
                $auditable = $request->input('auditable_type');

                // Si viene solo el nombre corto filtramos usando LIKE
                if (!str_contains($auditable, '\\')) {
                    $query->where('auditable_type', 'LIKE', "%{$auditable}%");
                } else {
                    // Si viene el FQCN completo
                    $query->where('auditable_type', $auditable);
                }
            }

            // Filtro por IP
            if ($request->filled('ip_address')) {
                $query->where('ip_address', 'LIKE', '%' . $request->input('ip_address') . '%');
            }

            // Filtro por rango de fechas (created_at)
            if ($request->filled('date_from')) {
                $query->whereDate('created_at', '>=', $request->input('date_from'));
            }

            if ($request->filled('date_to')) {
                $query->whereDate('created_at', '<=', $request->input('date_to'));
            }

            // ==========================
            // Límite por defecto
            // ==========================

            $tieneFiltros =
                $request->filled('event') ||
                $request->filled('user_id') ||
                $request->filled('user_type') ||
                $request->filled('auditable_type') ||
                $request->filled('ip_address') ||
                $request->filled('date_from') ||
                $request->filled('date_to');

            // Si NO hay ningún filtro, solo consideramos los últimos 250 registros
            if (!$tieneFiltros) {
                $query->limit(250);
            }

            // ==========================
            // Respuesta para DataTables
            // ==========================
            return DataTables::of($query)
                ->addIndexColumn()

                // Columna formateada de usuario
                ->addColumn('usuario', function ($row) {
                    $name = $row->user ? $row->user->nombres . ' ' . $row->user->apellido_pa : 'Sin nombre';
                    $id = $row->user_id ?: 'N/A';

                    return "{$name} (ID: {$id})";
                })

                // Columna de tipo_usuario que realizó la acción (user_type)
                ->addColumn('tipo_usuario', function ($row) {
                    if (!$row->user_type) {
                        return "Modelo no disponible";
                    }

                    return class_basename($row->user_type);
                })

                // Columna de modelo/tabla afectada (auditable_type + id)
                ->addColumn('modelo', function ($row) {
                    if (!$row->auditable) {
                        return "Modelo no disponible";
                    }
                    $modelo = class_basename($row->auditable_type);

                    return "{$modelo} (ID: {$row->auditable_id})";
                })

                // Columna de resumen de cambios (old_values vs new_values)
                ->addColumn('cambios', function ($row) {
                    $old = json_decode($row->old_values ?? '[]', true) ?: [];
                    $new = json_decode($row->new_values ?? '[]', true) ?: [];

                    // Si es un created, solo mostramos los valores nuevos
                    if ($row->event === 'created') {
                        if (empty($new)) {
                            return 'Registro creado.';
                        }

                        $resumen = [];
                        foreach ($new as $campo => $valorNuevo) {
                            $resumen[] = "{$campo}: {$valorNuevo}";
                        }

                        return implode("\n", array_slice($resumen, 0, 3)); // limitamos a 3 campos
                    }

                    // Si es deleted, mostramos los valores antiguos
                    if ($row->event === 'deleted') {
                        if (empty($old)) {
                            return 'Registro eliminado.';
                        }

                        $resumen = [];
                        foreach ($old as $campo => $valorAntiguo) {
                            $resumen[] = "{$campo}: {$valorAntiguo}";
                        }

                        return implode("\n", array_slice($resumen, 0, 3));
                    }

                    // Para updated, mostramos solo campos que cambiaron
                    $resumen = [];
                    foreach ($new as $campo => $valorNuevo) {
                        $valorAntiguo = $old[$campo] ?? null;

                        if ($valorAntiguo !== $valorNuevo) {
                            $valAnt = is_null($valorAntiguo) || $valorAntiguo === '' ? '—' : $valorAntiguo;
                            $valNue = is_null($valorNuevo) || $valorNuevo === '' ? '—' : $valorNuevo;
                            $resumen[] = "{$campo}: {$valAnt} → {$valNue}";
                        }
                    }

                    if (empty($resumen)) {
                        return 'Sin cambios detectados.';
                    }

                    return implode("\n", array_slice($resumen, 0, 3));
                })

                // Enviar los valores completos para el modal (como objetos JSON)
                ->addColumn('old_values_json', function ($row) {
                    // Devuelve un objeto/array decodificado; en JS podrás hacer JSON.stringify(row.old_values_json, null, 2)
                    $decoded = json_decode($row->old_values ?? '[]', true);
                    return $decoded ?: new \stdClass(); // si está vacío, devolvemos {} en lugar de []
                })

                ->addColumn('new_values_json', function ($row) {
                    $decoded = json_decode($row->new_values ?? '[]', true);
                    return $decoded ?: new \stdClass();
                })

                // Columna de fecha formateada (si quieres, si no lo haces en el front)
                ->editColumn('created_at', function ($row) {
                    // Ya viene como string "Y-m-d H:i:s", si quieres puedes dejarlo tal cual
                    return $row->created_at;
                })

                ->make(true);
        }

        return response()->json([
            'status'  => false,
            'mensaje' => 'Error: no se pueden cargar los registros de auditoría',
        ], 400);
    }
}
