// public/js/web/estadisticas.js

// --------- Helpers UI ---------
if (window.lucide && typeof lucide.createIcons === "function") {
    lucide.createIcons();
}

function scrollToSection(id) {
    const el = document.getElementById(id);
    if (!el) return;
    const y = el.getBoundingClientRect().top + window.pageYOffset - 80;
    window.scrollTo({ top: y, behavior: "smooth" });
}
window.scrollToSection = scrollToSection;

// Resalta botón activo en la barra lateral
let activeSection = "hero";
function updateActiveNavigation() {
    const sections = document.querySelectorAll("section[id]");
    let current = "hero";
    sections.forEach((s) => {
        const top = s.offsetTop,
            h = s.clientHeight;
        if (window.scrollY >= top - 200 && window.scrollY < top + h - 200) {
            current = s.id;
        }
    });
    if (current !== activeSection) {
        const prev = document.querySelector(`[data-section="${activeSection}"]`);
        if (prev) {
            prev.classList.remove("bg-[#E27227]");
            prev.classList.add("hover:bg-[#1A4FD3]/20");
        }
        const now = document.querySelector(`[data-section="${current}"]`);
        if (now) {
            now.classList.add("bg-[#E27227]");
            now.classList.remove("hover:bg-[#1A4FD3]/20");
        }
        activeSection = current;
    }
}
window.addEventListener("scroll", updateActiveNavigation);
window.addEventListener("load", updateActiveNavigation);

// --------- Datos desde el body ---------
const YEARS = JSON.parse(document.body.dataset.years || "[]");
const SECTIONS = JSON.parse(document.body.dataset.sections || "[]");

// --------- Paleta de colores ---------
const COLORS = [
    "#1A4FD3",
    "#4A84F7",
    "#E27227",
    "#00264B",
    "#10b981",
    "#ef4444",
    "#6366f1",
    "#f59e0b",
    "#0ea5e9",
    "#a855f7",
];

// --------- Funciones auxiliares ---------

// Filtra los años (columnas) cuya suma total es 0
function filterZeroYears(years, datasets) {
    const keepIdx = [];
    for (let i = 0; i < years.length; i++) {
        let sum = 0;
        datasets.forEach((d) => (sum += d.data?.[i] || 0));
        if (sum !== 0) keepIdx.push(i);
    }
    if (!keepIdx.length) return { years: [], datasets: [] };
    return {
        years: keepIdx.map((i) => years[i]),
        datasets: datasets.map((d) => ({
            ...d,
            data: keepIdx.map((i) => d.data?.[i] || 0),
        })),
    };
}

// Renderiza gráfico HD
function renderChartHD(ctx, config) {
    const canvas = ctx.canvas;
    const rect = canvas.getBoundingClientRect();
    const dpr = window.devicePixelRatio || 1;
    canvas.width = rect.width * dpr;
    canvas.height = rect.height * dpr;
    const c = canvas.getContext("2d");
    c.scale(dpr, dpr);
    return new Chart(ctx, config);
}

// --------- Construcción de gráficos ---------
function buildCharts() {
    if (!window.Chart) return;

    SECTIONS.forEach((sec, idx) => {
        const canvas = document.getElementById(`chart-${sec.id}`);
        if (!canvas) return;
        const ctx = canvas.getContext("2d");

        // Filtrar años sin datos
        let { years, datasets } = filterZeroYears(YEARS, sec.datasets);
        if (!years.length) {
            // Ocultar gráfico si no hay datos válidos
            canvas.closest(".stats-card").style.display = "none";
            return;
        }

        // Configurar datasets con colores
        const chartDatasets = datasets.map((d, i) => ({
            label: d.label,
            data: d.data,
            backgroundColor: COLORS[i % COLORS.length],
            borderColor: COLORS[i % COLORS.length],
            borderWidth: 1,
        }));

        // Renderizar en HD
        renderChartHD(ctx, {
            type: "bar",
            data: { labels: years, datasets: chartDatasets },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { position: "top" },
                    tooltip: {
                        backgroundColor: "#00264B",
                        titleColor: "#fff",
                        bodyColor: "#fff",
                    },
                },
                scales: {
                    x: {
                        ticks: {
                            color: "#00264B",
                            font: { size: 12 },
                        },
                        grid: { display: false },
                    },
                    y: {
                        beginAtZero: true,
                        ticks: {
                            color: "#00264B",
                            font: { size: 12 },
                        },
                        grid: { color: "#E5E7EB" },
                    },
                },
            },
        });
    });
}
window.addEventListener("load", buildCharts);

// --------- Scroll directo al gráfico ---------
function openAllDatasets(id) {
    const el = document.getElementById(`chart-${id}`);
    if (!el) return;
    const wrapper = el.closest(".stats-card");
    const y = wrapper.getBoundingClientRect().top + window.pageYOffset - 80;
    window.scrollTo({ top: y, behavior: "smooth" });
}
window.openAllDatasets = openAllDatasets;

// --------- Exportar tabla ---------
function exportTable(id) {
    const table = document.querySelector(`#table-${id} table`);
    if (!table) {
        alert("No se encontró la tabla de la sección.");
        return;
    }

    const rows = [...table.querySelectorAll("tr")];
    const csv = rows
        .map((row) => {
            const cols = [...row.querySelectorAll("th,td")];
            return cols
                .map((td) =>
                    `"${(td.textContent || "")
                        .trim()
                        .replace(/\r?\n|\r/g, " ")
                        .replace(/"/g, '""')}"`
                )
                .join(",");
        })
        .join("\n");

    const blob = new Blob(["\uFEFF" + csv], {
        type: "text/csv;charset=utf-8;",
    });
    const filename = `estadistica_${id}_${new Date()
        .toISOString()
        .slice(0, 10)}.csv`;

    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}
window.exportTable = exportTable;
