// public/js/admin/contactanos/redes/redes.js
document.addEventListener('DOMContentLoaded', () => {
  // ====== Util ======
  const csrf = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';

  // ====== Modales / Form ======
  const modalEditarEl = document.getElementById('modalEditar');
  const modalEditar = modalEditarEl ? new bootstrap.Modal(modalEditarEl) : null;

  const formEdit = document.getElementById('formEditar');
  const eNombre = document.getElementById('eNombre');
  const eEnlace = document.getElementById('eEnlace');
  const eIcono = document.getElementById('eIcono');
  const eActivo = document.getElementById('eActivo');

  // Vista previa de cono (opcional)
  const iconPreviewEdit = document.getElementById('iconPreviewEdit');
  const iconPreviewNew = document.getElementById('iconPreviewNew');
  const nIcono = document.getElementById('nIcono');
  if (nIcono && iconPreviewNew) {
    iconPreviewNew.className = nIcono.value || iconPreviewNew.className;
    nIcono.addEventListener('change', () => (iconPreviewNew.className = nIcono.value));
  }
  if (eIcono && iconPreviewEdit) {
    eIcono.addEventListener('change', () => (iconPreviewEdit.className = eIcono.value));
  }

  // ====== Helpers ======
  const swalToast = Swal.mixin({
    toast: true,
    position: 'top-end',
    showConfirmButton: false,
    timer: 2500,
    timerProgressBar: true,
  });

  // Actualiza una fila de la tabla con los nuevos valores (sin recargar)
  function patchRowInTable(url, payload) {
    // Encuentra la <tr> cuyo botn editar/eliminar apunta a la misma URL base (update/destroy)
    const rows = document.querySelectorAll('table.redes-table tbody tr');
    for (const tr of rows) {
      const btnEdit = tr.querySelector('.btn-edit');
      if (!btnEdit) continue;
      const editUrl = btnEdit.getAttribute('data-edit-url');
      if (editUrl === url) {
        // Actualiza celdas
        tr.querySelector('td:nth-child(2)').textContent = payload.nombre; // nombre
        const linkTd = tr.querySelector('td:nth-child(3) .truncate-url a');
        if (linkTd) {
          linkTd.textContent = payload.enlace;
          linkTd.setAttribute('href', payload.enlace);
        }
        const iconTd = tr.querySelector('td:nth-child(4) i');
        if (iconTd) iconTd.className = payload.icono;

        // Refresca data-* del botn editar
        btnEdit.setAttribute('data-nombre', payload.nombre);
        btnEdit.setAttribute('data-enlace', payload.enlace);
        btnEdit.setAttribute('data-icono', payload.icono);
        btnEdit.setAttribute('data-active', payload.is_active ? '1' : '0');

        break;
      }
    }
  }

  // ====== Editar: abrir modal y poblar ======
  document.querySelectorAll('.btn-edit').forEach(btn => {
    btn.addEventListener('click', () => {
      const nombre = btn.dataset.nombre || '';
      const enlace = btn.dataset.enlace || '';
      const icono  = btn.dataset.icono  || '';
      const active = parseInt(btn.dataset.active || '0', 10) === 1;

      // Setea campos
      eNombre.value = nombre;
      eEnlace.value = enlace;
      eIcono.value = icono;
      if (iconPreviewEdit) iconPreviewEdit.className = icono || '';
      eActivo.checked = active;

      // Setea action para spoofing
      formEdit.action = btn.dataset.editUrl; // VIENE DEL data-edit-url EN EL BLADE
      modalEditar && modalEditar.show();
    });
  });

  // ====== Editar: enviar (POST + _method=PUT para evitar 405) ======
  formEdit?.addEventListener('submit', async (e) => {
    e.preventDefault();
    const url = formEdit.action;
    const formData = new FormData();
    formData.append('_token', csrf);
    formData.append('_method', 'PUT'); // <-- spoofing
    formData.append('nombre', eNombre.value.trim());
    formData.append('enlace', eEnlace.value.trim());
    formData.append('icono', eIcono.value);
    formData.append('is_active', eActivo.checked ? '1' : '0');

    const submitBtn = formEdit.querySelector('button[type="submit"]');
    const original = submitBtn?.innerHTML;
    if (submitBtn) {
      submitBtn.disabled = true;
      submitBtn.innerHTML = `<span class="spinner-border spinner-border-sm me-2"></span>Guardando...`;
    }

    try {
      const res = await fetch(url, {
        method: 'POST',               // IMPORTANTE: POST para evitar 405
        headers: { 'Accept': 'application/json' },
        body: formData
      });

      if (!res.ok) {
        // Intenta leer errores de validacin 422
        let msg = 'No se pudo actualizar.';
        try {
          const data = await res.json();
          if (data?.errors) {
            const first = Object.values(data.errors)[0];
            msg = Array.isArray(first) ? first[0] : String(first);
          }
        } catch {}
        throw new Error(msg);
      }

      // Opcional: si backend devuelve JSON con el registro actualizado, salo; si no, arma con lo enviado
      let updated = { nombre: eNombre.value.trim(), enlace: eEnlace.value.trim(), icono: eIcono.value, is_active: eActivo.checked };
      try {
        const data = await res.json();
        updated = {
          nombre: data.nombre ?? updated.nombre,
          enlace: data.enlace ?? updated.enlace,
          icono:  data.icono  ?? updated.icono,
          is_active: ('is_active' in data) ? !!data.is_active : updated.is_active
        };
      } catch { /* si no hay JSON, seguimos con los valores del form */ }

      // Cierra modal y actualiza fila
      modalEditar && modalEditar.hide();
      patchRowInTable(url, updated);
      swalToast.fire({ icon: 'success', title: 'Cambios guardados' });
    } catch (err) {
      Swal.fire({ icon: 'error', title: 'Error', text: err.message || 'No se pudo actualizar.' });
    } finally {
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.innerHTML = original;
      }
    }
  });

  // ====== Eliminar con SweetAlert2 (POST + _method=DELETE para evitar 405) ======
  document.querySelectorAll('.btn-delete').forEach(btn => {
    btn.addEventListener('click', async (e) => {
      e.preventDefault();

      const title = btn.getAttribute('data-title') || 'este registro';
      const url   = btn.getAttribute('data-delete-url');

      const confirm = await Swal.fire({
        icon: 'warning',
        title: 'Eliminar',
        html: `Seguro que deseas eliminar <b>${title}</b>?`,
        showCancelButton: true,
        confirmButtonText: 'Si, eliminar',
        cancelButtonText: 'Cancelar'
      });

      if (!confirm.isConfirmed) return;

      const fd = new FormData();
      fd.append('_token', csrf);
      fd.append('_method', 'DELETE'); // <-- spoofing

      try {
        const res = await fetch(url, {
          method: 'POST',             // IMPORTANTE: POST para evitar 405
          headers: { 'Accept': 'application/json' },
          body: fd
        });

        if (!res.ok) {
          let msg = 'No se pudo eliminar.';
          try {
            const data = await res.json();
            if (data?.message) msg = data.message;
          } catch {}
          throw new Error(msg);
        }

        // Remueve la fila
        const tr = btn.closest('tr');
        if (tr) tr.remove();

        swalToast.fire({ icon: 'success', title: 'Registro eliminado' });
      } catch (err) {
        Swal.fire({ icon: 'error', title: 'Error', text: err.message || 'No se pudo eliminar.' });
      }
    });
  });

  // (Opcional) Mostrar modal "Nuevo" si vienes de validacin fallida del create
  if (window.redesShowNuevoModal) {
    const modalNuevoEl = document.getElementById('modalNuevo');
    modalNuevoEl && new bootstrap.Modal(modalNuevoEl).show();
  }

  // ====== Paginacin simple en cliente (tu misma lgica) ======
  const tableBody = document.querySelector('.redes-table tbody');
  const itemsPerPage = 5;
  let allRows = [];
  let currentPage = 1;

  function initPagination() {
    if (!tableBody) return;
    allRows = Array.from(tableBody.querySelectorAll('tr'));
    if (allRows.length <= itemsPerPage) return;
    createPaginationControls();
    showPage(1);
  }

  function showPage(page) {
    const startIndex = (page - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;

    allRows.forEach((row, index) => {
      row.style.display = (index >= startIndex && index < endIndex) ? '' : 'none';
    });

    currentPage = page;
    updatePaginationButtons();
  }

  function createPaginationControls() {
    const tableContainer = tableBody.parentElement.parentElement;
    const existing = document.querySelector('.redes-pagination');
    if (existing) existing.remove();

    const wrap = document.createElement('div');
    wrap.className = 'redes-pagination mt-3 d-flex justify-content-end';

    const totalPages = Math.ceil(allRows.length / itemsPerPage);
    let html = `
      <nav><ul class="pagination justify-content-end mb-0">
        <li class="page-item" id="redes-prev"><a class="page-link" href="#" aria-label="Previous"><span aria-hidden="true">&laquo;</span></a></li>
    `;
    for (let i = 1; i <= totalPages; i++) {
      html += `<li class="page-item ${i === 1 ? 'active' : ''}" data-page="${i}">
                 <a class="page-link" href="#" data-page="${i}">${i}</a>
               </li>`;
    }
    html += `
        <li class="page-item" id="redes-next"><a class="page-link" href="#" aria-label="Next"><span aria-hidden="true">&raquo;</span></a></li>
      </ul></nav>`;
    wrap.innerHTML = html;
    tableContainer.appendChild(wrap);

    document.getElementById('redes-prev')?.addEventListener('click', (e) => { e.preventDefault(); if (currentPage > 1) showPage(currentPage - 1); });
    document.getElementById('redes-next')?.addEventListener('click', (e) => { e.preventDefault(); const total = Math.ceil(allRows.length / itemsPerPage); if (currentPage < total) showPage(currentPage + 1); });
    document.querySelectorAll('.page-link[data-page]')?.forEach(a => a.addEventListener('click', (e) => { e.preventDefault(); showPage(parseInt(a.getAttribute('data-page'), 10)); }));
  }

  function updatePaginationButtons() {
    const totalPages = Math.ceil(allRows.length / itemsPerPage);
    const prevBtn = document.getElementById('redes-prev');
    const nextBtn = document.getElementById('redes-next');
    if (prevBtn) prevBtn.classList.toggle('disabled', currentPage <= 1);
    if (nextBtn) nextBtn.classList.toggle('disabled', currentPage >= totalPages);

    document.querySelectorAll('.page-item[data-page]')?.forEach(li => {
      const p = parseInt(li.getAttribute('data-page'), 10);
      li.classList.toggle('active', p === currentPage);
    });
  }

  initPagination();
});
