<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ProgramasConvenio;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;

class ProgramasConvenioController extends Controller
{
    /** Lista convenios por programa */
    public function index(int $programa)
    {
        $rows = ProgramasConvenio::where('programa_id', $programa)
            ->where(function ($q) { $q->whereNull('is_active')->orWhere('is_active', 1); })
            ->orderBy('id')
            ->get();

        return response()->json([
            'ok'        => true,
            'convenios' => $rows->map(fn($c) => [
                'id'            => $c->id,
                'entidad'       => $c->entidad,
                'imagen_url'    => $c->imagen_url,
                'documento_url' => $c->documento_url, 
                'enlace'        => $c->enlace, 
            ]),
        ]);
    }

    /** Mostrar un convenio */
    public function show(int $programa, int $convenio)
    {
        $c = ProgramasConvenio::where('programa_id', $programa)->findOrFail($convenio);

        return response()->json([
            'ok' => true,
            'convenio' => [
                'id'            => $c->id,
                'entidad'       => $c->entidad,
                'imagen_url'    => $c->imagen_url,
                'documento_url' => $c->documento_url, 
                'enlace'        => $c->enlace, 
            ],
        ]);
    }

    /** Crear convenio */
    public function store(Request $request, int $programa)
    {
        try {
            $data = $request->validate([
                'entidad'   => 'required|string|max:255',
                'imagen'    => 'nullable|image|max:40096',
                'documento' => 'nullable|mimes:pdf,doc,docx|max:90480',
                'enlace'    => 'nullable|string',
            ]);

            $imgPath = $this->saveImage($request->file('imagen'), null);
            $docPath = $this->saveDocument($request->file('documento'), null);

            $c = ProgramasConvenio::create([
                'programa_id' => $programa,
                'entidad'     => $data['entidad'],
                'imagen'      => $imgPath,
                'documento'   => $docPath,
                'enlace'      => $data['enlace'],
                'is_active'   => 1,
            ]);

            return response()->json(['ok' => true, 'id' => $c->id]);
        } catch (\Throwable $e) {
            \Log::error('Convenios store error', [
                'programa' => $programa,
                'msg' => $e->getMessage(),
                'file'=> $e->getFile(),
                'line'=> $e->getLine(),
            ]);
            return response()->json(['ok'=>false,'msg'=>'Error al crear convenio'], 500);
        }
    }

    /** Actualizar convenio (POST por compatibilidad con FormData) */
    public function update(Request $request, int $programa, int $convenio)
    {
        $c = ProgramasConvenio::where('programa_id', $programa)->findOrFail($convenio);

        $data = $request->validate([
            'entidad'   => 'required|string|max:255',
            'imagen'    => 'nullable|image|max:40096',
            'documento' => 'nullable|mimes:pdf,doc,docx|max:90480',
            'enlace'    => 'nullable|string',
        ]);

        $c->entidad = $data['entidad'];

        if ($request->filled('enlace')) {
            $c->enlace = $data['enlace'];
        }
        if ($request->hasFile('imagen')) {
            $c->imagen = $this->saveImage($request->file('imagen'), $c->imagen);
        }
        if ($request->hasFile('documento')) {
            $c->documento = $this->saveDocument($request->file('documento'), $c->documento);
        }

        $c->save();

        return response()->json(['ok' => true]);
    }

    /** Eliminar convenio (y archivos físicos si están en /images o /convenios) */
    public function destroy(int $programa, int $convenio)
    {
        $c = ProgramasConvenio::where('programa_id', $programa)->findOrFail($convenio);

        if ($c->imagen && str_starts_with($c->imagen, '/images/')) {
            $file = public_path(ltrim($c->imagen, '/'));
            if (is_file($file)) @unlink($file);
        }
        if ($c->documento && str_starts_with($c->documento, '/convenios/')) {
            $file = public_path(ltrim($c->documento, '/'));
            if (is_file($file)) @unlink($file);
        }

        $c->delete();

        return response()->json(['ok' => true]);
    }

    /** Guarda imagen en /public/images */
    private function saveImage(?UploadedFile $file, ?string $current): ?string
    {
        if (!$file) return $current;

        $dir = public_path('images');
        if (!is_dir($dir)) { @mkdir($dir, 0775, true); }

        $ext  = strtolower($file->getClientOriginalExtension() ?: 'jpg');
        $name = 'IMG_' . uniqid() . '.' . $ext;
        $file->move($dir, $name);

        if ($current && str_starts_with($current, '/images/')) {
            $old = public_path(ltrim($current, '/'));
            if (is_file($old)) @unlink($old);
        }

        return '/images/' . $name;
    }

    /** Guarda documento en /public/convenios */
    private function saveDocument(?UploadedFile $file, ?string $current): ?string
    {
        if (!$file) return $current;

        $dir = public_path('convenios'); // carpeta pública para documentos
        if (!is_dir($dir)) { @mkdir($dir, 0775, true); }

        $ext  = strtolower($file->getClientOriginalExtension() ?: 'pdf');
        $name = 'DOC_' . uniqid() . '.' . $ext;
        $file->move($dir, $name);

        if ($current && str_starts_with($current, '/convenios/')) {
            $old = public_path(ltrim($current, '/'));
            if (is_file($old)) @unlink($old);
        }

        return '/convenios/' . $name;
    }
}
