<?php

namespace App\Http\Controllers;

use App\Models\BecaBeneficiario;
use App\Models\ProgramasEstudio;
use App\Models\BecaTipo;
use Illuminate\Http\Request;
use App\Models\PeriodoAcademico;
use Illuminate\Support\Facades\DB; 

class BecaBeneficiarioController extends Controller
{
    
    public function index(Request $request)
    {
        $programas = ProgramasEstudio::where('is_active', 1)
            ->orderBy('nombre')->get(['id','nombre']);

        // Para poblar combos del modal Nuevo/Editar
        $periodos = PeriodoAcademico::orderBy('nombre')->get(['id','nombre','estado']);
        $tipos = BecaTipo::where('is_active', 1)
            ->orderBy('titulo')->get(['id','titulo','periodo_academico_id']);

        $programaId = $request->query('programa');

        $items = collect();
        if (!empty($programaId)) {
            // === JOINs como solicitaste ===
            $rows = DB::table('becas_beneficiario as bb')
                ->join('becas_tipo as bt', 'bb.tipo_beca_id', '=', 'bt.id')
                ->join('programas_estudios as pe', 'bb.programa_id', '=', 'pe.id')
                ->join('periodos_academicos as pa', 'bt.periodo_academico_id', '=', 'pa.id')
                ->where('bb.programa_id', $programaId)
                ->whereNull('bb.deleted_at')
                ->orderBy('bb.id', 'asc')
                ->select([
                    'bb.id',
                    'bb.programa_id',
                    'bb.tipo_beca_id',
                    'bb.nombre',
                    'bb.is_active',
                    'pe.nombre as programa_nombre',
                    'bt.titulo as tipo_titulo',
                    'bt.periodo_academico_id',
                    'pa.nombre as periodo_nombre',
                ])
                ->get();

            // === Adaptamos la estructura al shape que espera tu Blade ===
            $items = $rows->map(function ($r) {
                $o = new \stdClass();
                $o->id            = $r->id;
                $o->programa_id   = $r->programa_id;
                $o->tipo_beca_id  = $r->tipo_beca_id;
                $o->nombre        = $r->nombre;
                $o->is_active     = $r->is_active;

                // Para compatibilidad con $row->programa si algún día lo usas
                $o->programa = (object)[
                    'id'     => $r->programa_id,
                    'nombre' => $r->programa_nombre,
                ];

                // Lo que usa tu Blade:
                // $row->tipo->titulo
                // $row->tipo->periodo->nombre
                $o->tipo = (object)[
                    'titulo'               => $r->tipo_titulo,
                    'periodo_academico_id' => $r->periodo_academico_id,
                    'periodo'              => (object)[
                        'nombre' => $r->periodo_nombre,
                    ],
                ];

                return $o;
            });
        }

        return view('admin.admision_matricula.beca.beneficiarios.index', compact(
            'programas','periodos','tipos','items','programaId'
        ));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'programa_id'  => ['required','integer','exists:programas_estudios,id'],
            'tipo_beca_id' => ['required','integer','exists:becas_tipo,id'],
            'nombre'       => ['required','string','max:150'],
            'is_active'    => ['nullable','boolean'],
        ]);

        BecaBeneficiario::create([
            'programa_id'  => $data['programa_id'],
            'tipo_beca_id' => $data['tipo_beca_id'],
            'nombre'       => $data['nombre'],
            'is_active'    => $request->boolean('is_active'),
        ]);

        return back()->with('success','Beneficiario creado.');
    }

    public function update(Request $request, BecaBeneficiario $beneficiario)
    {
        $data = $request->validate([
            'programa_id'  => ['required','integer','exists:programas_estudios,id'],
            'tipo_beca_id' => ['required','integer','exists:becas_tipo,id'],
            'nombre'       => ['required','string','max:150'],
            'is_active'    => ['nullable','boolean'],
        ]);

        $beneficiario->update([
            'programa_id'  => $data['programa_id'],
            'tipo_beca_id' => $data['tipo_beca_id'],
            'nombre'       => $data['nombre'],
            'is_active'    => $request->boolean('is_active'),
        ]);

        return back()->with('success','Beneficiario actualizado.');
    }

    public function destroy(BecaBeneficiario $beneficiario)
    {
        $beneficiario->delete(); // SoftDelete
        return back()->with('success','Beneficiario eliminado.');
    }
}
