<?php

namespace App\Http\Controllers\docente;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Studentgrade;
use Illuminate\Support\Facades\Validator;
use \Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Collection;
use Yajra\DataTables\Facades\DataTables;
use App\Models\Teacher;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ZoomController extends Controller
{
    /**
     * Obtiene el token de acceso utilizando el flujo Server-to-Server OAuth.
     */
    private function getAccessToken()
    {
        $url = 'https://zoom.us/oauth/token';

        // Obtener el usuario autenticado
        $user = Auth::user();

        if (!$user) {
            return null;  // Devolver null si el usuario no está autenticado
        }

        // Acceder a la relación 'teachers'
        $teacherdata = $user->teachers;

        if (!$teacherdata || $teacherdata->isEmpty()) {
            return null;  // Devolver null si no hay profesores asociados
        }

        $idteacher = $teacherdata[0]->id;

        // Buscar la información del profesor en la base de datos
        $data = Teacher::find($idteacher);

        if (!$data) {
            return null;  // Devolver null si no se encuentra al profesor
        }

        // Validar las credenciales del profesor
        $accountId = $data->account_id;
        $clientId = $data->client_id;
        $clientSecret = $data->client_secret;

        if (is_null($accountId) || is_null($clientId) || is_null($clientSecret)) {
            return null;  // Devolver null si faltan credenciales
        }

        // Crear la cabecera de autorización Basic Auth
        $authorization = base64_encode("{$clientId}:{$clientSecret}");

        // Enviar la solicitud a Zoom
        $response = Http::withHeaders([
            'Authorization' => "Basic $authorization",
        ])->asForm()->post($url, [
            'grant_type' => 'account_credentials',
            'account_id' => $accountId,
        ]);

        // Verificar si la respuesta es exitosa y devolver el token
        if ($response->successful()) {
            return $response->json()['access_token'];
        }

        return null;  // Si no se obtiene el token, devolver null
    }

    /**
     * Lista las reuniones del usuario.
     */
    public function listMeetings()
    {
        // Obtiene el token de acceso
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return redirect()->route('docente.zoom.index')->with('error', 'No se pudo obtener el token de acceso.');
        }

        // Realiza la solicitud para obtener las reuniones
        $response = Http::withToken($accessToken)
            ->get('https://api.zoom.us/v2/users/me/meetings');

        // Verifica si la respuesta fue exitosa
        if ($response->successful()) {
            // Intenta obtener las reuniones
            $meetings = $response->json()['meetings'] ?? [];

            // Si no hay reuniones, muestra un mensaje adecuado
            if (empty($meetings)) {
                return view('docente.zoom.index', ['meetings' => []])->with('error', 'No tienes reuniones programadas.');
            }

            return view('zoom.meetings', compact('meetings'));
        }

        // Si hubo un error al obtener las reuniones
        return redirect()->route('docente.zoom.index')->with('error', 'No se pudieron obtener las reuniones de Zoom');
    }

    /**
     * Crea una nueva reunión.
     */
    public function createMeeting(Request $request)
    {
        // Establecer la zona horaria explícita
        $timezone = 'America/Lima';
        $now = Carbon::now($timezone); // Fecha y hora actual en Lima
        $startTime = Carbon::parse($request->input('start_time'), $timezone); // Fecha elegida por el usuario

        // Validar que no se programen reuniones en el pasado
        if ($startTime->lessThan($now)) {
            return redirect()->back()->with('error', 'No puedes programar una reunión en el pasado. La fecha y hora seleccionada es anterior al momento actual.');
        }

        // Validar horas en el día actual
        if ($startTime->isToday() && $startTime->lessThan($now)) {
            return redirect()->back()->with('error', 'No puedes programar una reunión en una hora pasada para el día actual. Elige una hora futura.');
        }

        // Crear la reunión en Zoom
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return redirect()->route('docente.zoom.index')->with('error', 'No se pudo obtener el token de acceso de Zoom. Intenta nuevamente más tarde.');
        }
        $response = Http::withToken($accessToken)
            ->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic' => $request->input('topic', 'Nueva reunión'),
                'type' => 2,
                'start_time' => $startTime->toIso8601String(),
                'duration' => $request->input('duration', 60),
                'timezone' => $timezone,
            ]);

        if ($response->successful()) {
            return redirect()->route('docente.zoom.index')->with('success', 'Reunión creada exitosamente.');
        }

        // Si Zoom responde con error
        return redirect()->back()->with('error', 'Hubo un problema al crear la reunión en Zoom. Intenta nuevamente.');
    }

    /**
     * Elimina una reunión por ID.
     */
    public function deleteMeeting($meetingId)
    {
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return redirect()->route('docente.zoom.index')->with('error', 'No se pudo obtener el token de acceso.');
        }

        $response = Http::withToken($accessToken)
            ->delete("https://api.zoom.us/v2/meetings/{$meetingId}");

        if ($response->successful()) {
            return redirect()->route('docente.zoom.index')->with('success', 'Reunión eliminada exitosamente.');
        }

        return redirect()->back()->with('error', 'No se pudo eliminar la reunión.');
    }

    public function showMeetingsPage(Request $request)
    {
        if (Auth::check()) {
            $userTypeAccesocombo = $request->session()->get('userTypeAccesocombo');
            if ($userTypeAccesocombo == "2") {
                //si el usuario esta autentificado es docente o estudiante no le pasamos la lista del sidebar
                $tipoacceso = 2;
                $accessToken = $this->getAccessToken();

                if (!$accessToken) {
                    return view('docente.zoom.index', ['meetings' => []])
                        ->with('error', 'No se pudo obtener el token de acceso.')->with('datatipoacceso', $tipoacceso);
                }

                $response = Http::withToken($accessToken)
                    ->get('https://api.zoom.us/v2/users/me/meetings');

                if ($response->successful()) {
                    $meetings = $response->json()['meetings'] ?? [];
                    return view('docente.zoom.index', compact('meetings'))->with('datatipoacceso', $tipoacceso);
                }

                return view('docente.zoom.index', ['meetings' => []])
                    ->with('error', 'No se pudieron obtener las reuniones de Zoom.')->with('datatipoacceso', $tipoacceso);
            } else if ($userTypeAccesocombo == "1") {
                //si el usuario esta autentificado es docente o estudiante no le pasamos la lista del sidebar
                return redirect()->route('home.adm.dashboard')->withSuccess('Opps! You do not have access');
            } else if ($userTypeAccesocombo == "3") {
                //si el usuario esta autentificado es docente o estudiante no le pasamos la lista del sidebar
                return redirect()->route('home.student.index')->withSuccess('Opps! You do not have access');
            }
        } else {
            return redirect()->route('login')->withSuccess('Opps! You do not have access');
        }
    }

    public function getFilteredMeetings(Request $request)
    {
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 500);
        }

        $response = Http::withToken($accessToken)->get('https://api.zoom.us/v2/users/me/meetings');

        if ($response->successful()) {
            $meetings = $response->json()['meetings'] ?? [];
            $now = Carbon::now();

            $filteredMeetings = collect($meetings)->filter(function ($meeting) use ($now, $request) {
                $startTime = Carbon::parse($meeting['start_time']);
                return $request->type === 'upcoming'
                    ? $startTime->isAfter($now)
                    : $startTime->isBefore($now);
            });

            return response()->json($filteredMeetings->values());
        }

        return response()->json(['error' => 'No se pudieron obtener las reuniones de Zoom.'], 500);
    }

    public function deleteMeetingAndDatabase($meetingId, Request $request)
    {
        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 401);
        }

        // Eliminar reunión en Zoom
        $response = Http::withToken($accessToken)
            ->delete("https://api.zoom.us/v2/meetings/{$meetingId}");

        if ($response->successful()) {
            // Actualizar base de datos (elimino solo el id_zoom_meeting)
            DB::table('clases_asignaturas')
                ->where('id', $request->input('idClass'))
                ->update(['id_zoom_meeting' => null]);

            return response()->json(['success' => 'Reunión eliminada correctamente.']);
        }

        return response()->json(['error' => 'No se pudo eliminar la reunión.'], 500);
    }

    public function getLinkMeetingZoom(Request $request, $id_zoom_meeting)
    {
        // Obtiene el token de acceso
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 401);
        }

        // Realiza la solicitud para obtener los detalles de la reunión
        $response = Http::withToken($accessToken)
            ->get("https://api.zoom.us/v2/meetings/$id_zoom_meeting");

        // Verifica si la respuesta fue exitosa
        if ($response->successful()) {
            $meeting = $response->json();

            // Verifica si la reunión tiene un join_url
            if (isset($meeting['join_url'])) {
                return response()->json(['join_url' => $meeting['join_url']]);
            } else {
                return response()->json(['error' => 'No se pudo obtener el link de la reunión.'], 404);
            }
        }

        // Si hubo un error al obtener la reunión
        return response()->json(['error' => 'No se encontró la reunión de Zoom'], 404);
    }

    public function createZoomMeetingForClass($zoomData)
    {
        // Obtiene el token de acceso
        $accessToken = $this->getAccessToken();

        if (!$accessToken) {
            return response()->json(['error' => 'No se pudo obtener el token de acceso.'], 401);
        }

        // Crear la reunión en Zoom
        $response = Http::withToken($accessToken)
            ->post('https://api.zoom.us/v2/users/me/meetings', [
                'topic' => "Clase: {$zoomData->dia} - {$zoomData->horaini}",
                'type' => 2,  // Tipo 2 para reuniones programadas
                'start_time' => $zoomData->startTime->toIso8601String(),
                'duration' => $zoomData->duracion, // Duración de la clase
                'timezone' => $zoomData->timezone,
                'agenda' => $zoomData->agenda,
            ]);

        if ($response->successful()) {
            // Guardar el ID de la reunión en la base de datos si es necesario
            $meeting = $response->json();
            DB::table('clases_asignaturas')->where('id', $zoomData->idclass)->update([
                'id_zoom_meeting' => $meeting['id'],
            ]);

            return response()->json(['success' => 'Reunión creada exitosamente.']);
        }

        return response()->json(['error' => 'Hubo un problema al crear la reunión en Zoom.']);
    }
}
