document.addEventListener('DOMContentLoaded', function() {
    if (typeof lucide !== 'undefined') {
        lucide.createIcons();
    }

    let allTableRows = [];
    let filteredRows = [];
    let currentPage = 1;
    const itemsPerPage = 5;

    function initializeTableRows() {
        const tableBody = document.getElementById('table-body');
        if (tableBody) {
            allTableRows = Array.from(tableBody.querySelectorAll('.table-row'));
            filteredRows = [...allTableRows]; // por defecto, todos visibles
        }
        return allTableRows.length;
    }

    function showPage(page) {
        const startIndex = (page - 1) * itemsPerPage;
        const endIndex = startIndex + itemsPerPage;

        // Ocultar todas primero
        allTableRows.forEach(row => (row.style.display = 'none'));

        // Mostrar solo las filas filtradas correspondientes a la página actual
        filteredRows.slice(startIndex, endIndex).forEach(row => (row.style.display = ''));

        currentPage = page;
        updatePaginationButtons();
    }

    // Crear controles de paginación
    function createPaginationControls() {
        const tableContainer = document.querySelector('.bg-white.rounded-3xl');
        if (!tableContainer) return;

        const existingPagination = document.querySelector('.tupa-pagination');
        if (existingPagination) existingPagination.remove();

        const totalPages = Math.ceil(filteredRows.length / itemsPerPage);
        if (totalPages <= 1) return;

        const paginationDiv = document.createElement('div');
        paginationDiv.className = 'tupa-pagination flex justify-center items-center gap-2 py-6 flex-wrap';

        // Botón anterior
        const prevBtn = document.createElement('button');
        prevBtn.textContent = '←';
        prevBtn.className = 'px-3 py-2 text-sm font-medium border border-gray-300 rounded-lg hover:bg-gray-100 disabled:opacity-50';
        prevBtn.addEventListener('click', () => {
            if (currentPage > 1) showPage(currentPage - 1);
        });
        paginationDiv.appendChild(prevBtn);

        // Botones numerados
        for (let i = 1; i <= totalPages; i++) {
            const btn = document.createElement('button');
            btn.textContent = i;
            btn.className = 'page-btn px-3 py-2 text-sm font-medium border border-gray-300 rounded-lg hover:bg-[#1A4FD3] hover:text-white transition-colors';
            btn.addEventListener('click', () => showPage(i));
            paginationDiv.appendChild(btn);
        }

        // Botón siguiente
        const nextBtn = document.createElement('button');
        nextBtn.textContent = '→';
        nextBtn.className = 'px-3 py-2 text-sm font-medium border border-gray-300 rounded-lg hover:bg-gray-100 disabled:opacity-50';
        nextBtn.addEventListener('click', () => {
            if (currentPage < totalPages) showPage(currentPage + 1);
        });
        paginationDiv.appendChild(nextBtn);

        tableContainer.appendChild(paginationDiv);
        updatePaginationButtons();
    }

    function updatePaginationButtons() {
        const totalPages = Math.ceil(filteredRows.length / itemsPerPage);
        const pageButtons = document.querySelectorAll('.page-btn');

        pageButtons.forEach((btn, i) => {
            if (i + 1 === currentPage) {
                btn.classList.add('bg-[#1A4FD3]', 'text-white');
                btn.classList.remove('hover:bg-[#1A4FD3]');
            } else {
                btn.classList.remove('bg-[#1A4FD3]', 'text-white');
            }
        });

        const prevBtn = document.querySelector('.tupa-pagination button:first-child');
        const nextBtn = document.querySelector('.tupa-pagination button:last-child');

        if (prevBtn && nextBtn) {
            prevBtn.disabled = currentPage === 1;
            nextBtn.disabled = currentPage === totalPages;
        }
    }

    function refreshPagination() {
        const pagination = document.querySelector('.tupa-pagination');
        if (pagination) pagination.remove();
        createPaginationControls();
        showPage(1);
    }

    // === FILTRO DE BÚSQUEDA ===
    const searchInput = document.getElementById('tupa-search');
    const noResults = document.getElementById('no-results');

    if (searchInput) {
        searchInput.addEventListener('input', function() {
            const query = this.value.trim().toLowerCase();

            if (query === '') {
                // Restaurar todas las filas
                filteredRows = [...allTableRows];
                if (noResults) noResults.classList.add('hidden');
                refreshPagination();
                return;
            }

            // Filtrar filas
            filteredRows = allTableRows.filter(row => {
                const conceptoCell = row.querySelector('.concepto-cell');
                const text = conceptoCell ? conceptoCell.textContent.toLowerCase() : '';
                return text.includes(query);
            });

            if (noResults) {
                noResults.classList.toggle('hidden', filteredRows.length > 0);
            }

            refreshPagination();
        });
    }

    setTimeout(() => {
        initializeTableRows();
        refreshPagination();
    }, 100);
});
