document.addEventListener("DOMContentLoaded", () => {
  const CSRF_TOKEN = document.querySelector('meta[name="csrf-token"]').getAttribute("content");

  const tbody = document.getElementById("tupaBody");
  const searchInput = document.getElementById("searchInput");
  const alertBox = document.getElementById("alertContainer");
  const paginationContainer = document.getElementById("paginationContainer");

  const modalNuevo  = new bootstrap.Modal(document.getElementById("modalNuevo"));
  const modalEditar = new bootstrap.Modal(document.getElementById("modalEditar"));
  const modalEliminarEl = document.getElementById("modalEliminar");

  const formNuevo  = document.getElementById("formNuevo");
  const formEditar = document.getElementById("formEditar");
  const formEliminar = document.getElementById("formEliminar");

  // Campos (ajuste de labels / tipos)
  const createNombre = document.getElementById("createConcepto");
  const createDesc   = document.getElementById("createMonto");
  const editId       = document.getElementById("editId");
  const editNombre   = document.getElementById("editConcepto");
  const editDesc     = document.getElementById("editMonto");
  const editActive   = document.getElementById("editActive");

  // Mostrar "Nombre" y "Descripción" (ocultamos switch si no se usa)
  ["createActive", "editActive"].forEach((id) => {
    const el = document.getElementById(id);
    if (el && el.closest(".form-check")) el.closest(".form-check").style.display = "none";
  });
  document.querySelectorAll("#modalNuevo label, #modalEditar label").forEach((l) => {
    if (l.htmlFor.includes("Concepto")) l.textContent = "Nombre *";
    if (l.htmlFor.includes("Monto"))    l.textContent = "Descripción *";
  });
  createDesc.type = "text";
  editDesc.type   = "text";

  let rows = [];
  let currentPage = 1;
  const itemsPerPage = 5;

  const notify = (type, msg) => {
    alertBox.style.display = "block";
    alertBox.innerHTML = `
      <div class="alert alert-${type} alert-dismissible fade show mb-0">
        ${msg}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
      </div>`;
    setTimeout(() => (alertBox.style.display = "none"), 3000);
  };

  const esc = (s) => String(s ?? '')
    .replace(/&/g, '&amp;').replace(/"/g, '&quot;')
    .replace(/</g, '&lt;').replace(/>/g, '&gt;');

  function renderTable(term = "", page = 1) {
    const q = term.trim().toLowerCase();
    const data = q
      ? rows.filter(r =>
          String(r.id).includes(q) ||
          (r.nombre || "").toLowerCase().includes(q) ||
          (r.descripcion || "").toLowerCase().includes(q)
        )
      : rows;

    if (!data.length) {
      tbody.innerHTML = `<tr><td colspan="4" class="text-center text-muted">Sin registros.</td></tr>`;
      paginationContainer.innerHTML = "";
      return;
    }

    // Paginación correcta
    const totalPages = Math.ceil(data.length / itemsPerPage) || 1;
    currentPage = Math.max(1, Math.min(page, totalPages));
    const start = (currentPage - 1) * itemsPerPage;
    const end   = start + itemsPerPage;
    const pageData = data.slice(start, end);

    tbody.innerHTML = "";
    for (const r of pageData) {
      const tr = document.createElement("tr");
      tr.innerHTML = `
        <td class="col-id">${r.id}</td>
        <td class="text-muted fw-semibold col-nombre">${esc(r.nombre)}</td>
        <td class="text-muted col-descripcion">${esc(r.descripcion)}</td>
        <td class="text-center col-acciones">
          <div class="d-inline-flex gap-1">
            <button class="btn btn-warning btn-sm text-white"
                    data-action="edit" data-id="${r.id}" title="Editar">
              <i class="fa-regular fa-pen-to-square"></i>
            </button>
            <button class="btn btn-danger btn-sm btn-delete"
                    data-id="${r.id}"
                    data-title="${esc(r.nombre)}"
                    data-bs-toggle="modal"
                    data-bs-target="#modalEliminar"
                    title="Eliminar">
              <i class="fa-regular fa-trash-can"></i>
            </button>
          </div>
        </td>`;
      tbody.appendChild(tr);
    }

    renderPagination(totalPages, currentPage, term);
  }

  function renderPagination(totalPages, currentPage, term) {
    let html = `<nav><ul class="pagination mb-0">`;
    html += `<li class="page-item${currentPage === 1 ? " disabled" : ""}">
      <a class="page-link" href="#" data-page="${currentPage - 1}">&laquo;</a></li>`;
    for (let i = 1; i <= totalPages; i++) {
      html += `<li class="page-item${i === currentPage ? " active" : ""}">
        <a class="page-link" href="#" data-page="${i}">${i}</a></li>`;
    }
    html += `<li class="page-item${currentPage === totalPages ? " disabled" : ""}">
      <a class="page-link" href="#" data-page="${currentPage + 1}">&raquo;</a></li>`;
    html += `</ul></nav>`;
    paginationContainer.innerHTML = html;

    paginationContainer.querySelectorAll(".page-link[data-page]").forEach((link) => {
      link.addEventListener("click", function (e) {
        e.preventDefault();
        const page = parseInt(this.getAttribute("data-page"));
        if (page >= 1 && page <= totalPages) {
          renderTable(term || searchInput.value, page);
        }
      });
    });
  }

  async function fetchGrid() {
    const res = await fetch(window.TIPO_RECLAMACION_ROUTES.grid);
    if (!res.ok) {
      notify("danger", "Error al cargar la lista");
      return;
    }
    rows = await res.json();
    renderTable(searchInput.value || "", 1);
  }

  searchInput.addEventListener("input", (e) => renderTable(e.target.value, 1));

  // Editar / preparar eliminar
  tbody.addEventListener("click", (e) => {
    const btn = e.target.closest("button");
    if (!btn) return;

    const id = btn.dataset.id;

    if (btn.dataset.action === "edit") {
      const r = rows.find((x) => String(x.id) === String(id));
      if (!r) return;
      editId.value     = r.id;
      editNombre.value = r.nombre ?? "";
      editDesc.value   = r.descripcion ?? "";
      editActive.checked = !!r.is_active;
      modalEditar.show();
    }
    // El "delete" solo abre el modal eliminar, la URL se arma abajo (show.bs.modal)
  });

  // Armar acción del form eliminar al abrir el modal
  modalEliminarEl.addEventListener('show.bs.modal', (ev) => {
    const trigger = ev.relatedTarget;
    const id    = trigger?.getAttribute('data-id');
    const title = trigger?.getAttribute('data-title') || 'este registro';

    const urlTpl = (window.TIPO_RECLAMACION_ROUTES?.destroy || '');
    const url    = urlTpl.replace(':id', id);

    // set action + título dentro del modal
    formEliminar.setAttribute('action', url);
    const delTitle = document.getElementById('delTitulo');
    if (delTitle) delTitle.textContent = title;
  });

  // INTERCEPTAR submit del form de eliminar -> fetch DELETE (evita navegar a JSON)
  formEliminar.addEventListener('submit', async (e) => {
    e.preventDefault();
    const action = formEliminar.getAttribute('action');
    if (!action) return;

    const submitBtn = formEliminar.querySelector('button[type="submit"]');
    const original  = submitBtn?.textContent || 'Sí, eliminar';
    if (submitBtn) {
      submitBtn.disabled = true;
      submitBtn.textContent = 'Eliminando...';
    }

    try {
      const res = await fetch(action, {
        method: 'DELETE',
        headers: {
          'X-CSRF-TOKEN': CSRF_TOKEN,
          'Accept': 'application/json'
        }
      });
      if (!res.ok) {
        let msg = 'No se pudo eliminar.';
        try {
          const data = await res.json();
          if (data?.message) msg = data.message;
        } catch {}
        throw new Error(msg);
      }

      // Cerrar modal y refrescar grid sin salir de la página
      const inst = bootstrap.Modal.getInstance(modalEliminarEl);
      inst?.hide();
      await fetchGrid();
      notify('success', 'Registro eliminado.');

    } catch (err) {
      notify('danger', err?.message || 'Error al eliminar.');
      console.error('[DELETE tipo-reclamacion]', err);
    } finally {
      if (submitBtn) {
        submitBtn.disabled = false;
        submitBtn.textContent = original;
      }
    }
  });

  // Crear
  formNuevo.addEventListener("submit", async (e) => {
    e.preventDefault();
    const payload = {
      nombre: createNombre.value.trim(),
      descripcion: createDesc.value.trim(),
      is_active: 1
    };
    const res = await fetch(window.TIPO_RECLAMACION_ROUTES.store, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-CSRF-TOKEN": CSRF_TOKEN,
        "Accept": "application/json",
      },
      body: JSON.stringify(payload),
    });
    if (res.ok) {
      formNuevo.reset();
      modalNuevo.hide();
      await fetchGrid();
      notify("success", "Registro creado correctamente.");
    } else {
      notify("danger", "Error al crear.");
    }
  });

  // Guardar edición
  formEditar.addEventListener("submit", async (e) => {
    e.preventDefault();
    const id = editId.value;
    const payload = {
      nombre: editNombre.value.trim(),
      descripcion: editDesc.value.trim(),
      is_active: editActive.checked ? 1 : 0
    };
    const res = await fetch(window.TIPO_RECLAMACION_ROUTES.update.replace(":id", id), {
      method: "PUT",
      headers: {
        "Content-Type": "application/json",
        "X-CSRF-TOKEN": CSRF_TOKEN,
        "Accept": "application/json",
      },
      body: JSON.stringify(payload),
    });
    if (res.ok) {
      modalEditar.hide();
      await fetchGrid();
      notify("success", "Cambios guardados.");
    } else {
      notify("danger", "Error al actualizar.");
    }
  });

  // Carga inicial
  fetchGrid();
});
