<?php
// app/Http/Controllers/TupaController.php
namespace App\Http\Controllers;

use App\Models\Tupa;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;

class TupaController extends Controller
{
    public function index()
    {
        $items = Tupa::orderBy('id')->get();

        return view('admin.transparencia.tupa.index', compact('items'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'concepto'  => ['required','string','max:255'],
            'monto'     => ['required','numeric','min:0','max:999999.99'],
            'is_active' => ['nullable','boolean'],
        ]);

        Tupa::create([
            'concepto'  => $data['concepto'],
            'monto'     => $data['monto'],
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()->route('tupa.index')->with('success', 'Registro creado.');
    }

    public function update(Request $request, Tupa $tupa)
    {
        $data = $request->validate([
            'concepto'  => ['required','string','max:255'],
            'monto'     => ['required','numeric','min:0','max:999999.99'],
            'is_active' => ['nullable','boolean'],
        ]);

        $tupa->update([
            'concepto'  => $data['concepto'],
            'monto'     => $data['monto'],
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()->route('tupa.index')->with('success', 'Registro actualizado.');
    }

    public function destroy(Tupa $tupa)
    {
        $tupa->delete(); // Soft delete
        return redirect()->route('tupa.index')->with('success', 'Registro eliminado.');
    }

    /** Actualizar convenio (POST por compatibilidad con FormData) */
    public function updateDocument(Request $request)
    {
        $request->validate([
            'documento' => 'nullable|mimes:pdf,doc,docx|max:90480',
        ]);

        if ($request->hasFile('documento')) {
            $this->saveDocument($request->file('documento'), null);
        }

        return response()->json(['ok' => true, 'message' => 'El documento TUPA fue actualizado correctamente.']);
    }

    /** Guarda documento en /public/convenios */
    private function saveDocument(?UploadedFile $file, ?string $current)
    {
        if (!$file) return $current;

        $dir = public_path('assets'); // carpeta pública para documentos
        if (!is_dir($dir)) { @mkdir($dir, 0775, true); }

        $ext  = strtolower($file->getClientOriginalExtension() ?: 'pdf');
        $name = 'TUPA.' . $ext;
        $file->move($dir, $name);

        if ($current && str_starts_with($current, '/assets/')) {
            $old = public_path(ltrim($current, '/'));
            if (is_file($old)) @unlink($old);
        }
    }
}
