<?php

namespace App\Http\Controllers;

use App\Models\DocumentoGestion;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class DocumentoGestionController extends Controller
{
    public function index()
    {
        $items = DocumentoGestion::orderBy('id')->get();
        return view('admin.transparencia.documentos.index', compact('items'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'nombre'      => ['required', 'string', 'max:200'],
            'descripcion' => ['required', 'string', 'max:2000'],
            'documento'   => ['required', 'file', 'mimes:pdf', 'max:90240'],
            'is_active'   => ['nullable', 'boolean'],
        ]);

        if ($request->hasFile('documento')) {
            $file = $request->file('documento');
            $name = Str::uuid() . '_' .
                    Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) .
                    '.' . $file->getClientOriginalExtension();

            $file->move(public_path('assets'), $name);
            $data['documento'] = 'assets/' . $name;
        }

        $data['is_active'] = $request->boolean('is_active');

        DocumentoGestion::create($data);

        return back()->with('success', 'Documento creado.');
    }

    // ⬇️ Cambia $doc -> $documento para coincidir con {documento}
    public function update(Request $request, DocumentoGestion $documento)
    {
        $data = $request->validate([
            'nombre'      => ['required', 'string', 'max:200'],
            'descripcion' => ['required', 'string', 'max:2000'],
            'documento'   => ['nullable', 'file', 'mimes:pdf', 'max:90240'],
            'is_active'   => ['nullable', 'boolean'],
        ]);

        if ($request->hasFile('documento')) {
            if ($documento->documento && file_exists(public_path($documento->documento))) {
                @unlink(public_path($documento->documento));
            }

            $file = $request->file('documento');
            $name = Str::uuid() . '_' .
                    Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) .
                    '.' . $file->getClientOriginalExtension();

            $file->move(public_path('assets'), $name);
            $data['documento'] = 'assets/' . $name;
        }

        $data['is_active'] = $request->boolean('is_active');

        $documento->update($data);

        return back()->with('success', 'Documento actualizado.');
    }

    // ⬇️ Cambia $doc -> $documento para coincidir con {documento}
    public function destroy(DocumentoGestion $documento)
    {
        if ($documento->documento && file_exists(public_path($documento->documento))) {
            @unlink(public_path($documento->documento));
        }

        $documento->delete();

        return back()->with('success', 'Documento eliminado.');
    }

    public function file(DocumentoGestion $documento)
    {
        // (Si usas public/assets no necesitas Storage aquí; lo puedes servir con asset().)
        abort_unless($documento->documento && file_exists(public_path($documento->documento)), 404);
        return response()->file(public_path($documento->documento));
    }
}
